package com.elitescloud.boot.tenant.client.common;

import com.elitescloud.boot.tenant.client.support.database.TableCreate;
import com.elitescloud.boot.tenant.client.support.database.mysql.MySqlExporter;
import com.elitescloud.boot.tenant.client.support.database.oracle.OracleExporter;
import org.springframework.util.Assert;

import java.util.function.Consumer;

/**
 * 数据库导出工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2022/5/27
 */
public class DataBaseExport {

    /**
     * jdbc连接
     */
    private String jdbcUrl;
    /**
     * 驱动名称
     */
    private String driverClassName;
    /**
     * 用户名
     */
    private String username;
    /**
     * 密码
     */
    private String password;
    /**
     * 要导出的表
     */
    private String[] tables;
    /**
     * 创建策略
     */
    private TableCreate tableCreate = TableCreate.CREATE;

    /**
     * 表名大小写敏感
     */
    private boolean tableNameSensitive = false;

    /**
     * 列名大小写敏感
     */
    private boolean columnNameSensitive = false;

    protected DataBaseExport() {
    }

    /**
     * 构建导出工具类
     *
     * @return 导出工具类
     */
    public static DataBaseExport builder() {
        return new DataBaseExport();
    }

    /**
     * 导出MySQL表结构
     *
     * @param tableDdlConsumer 表结构ddl
     */
    public void exportToMysql(Consumer<String> tableDdlConsumer) {
        new MySqlExporter(this).export(tableDdlConsumer);
    }

    /**
     * 导出Oracle表结构
     *
     * @param tableDdlConsumer 表结构ddl
     */
    public void exportToOracle(Consumer<String> tableDdlConsumer) {
        new OracleExporter(this).export(tableDdlConsumer);
    }

    public DataBaseExport setJdbcUrl(String jdbcUrl) {
        this.jdbcUrl = jdbcUrl;
        return this;
    }

    public DataBaseExport setDriverClassName(String driverClassName) {
        this.driverClassName = driverClassName;
        return this;
    }

    public DataBaseExport setUsername(String username) {
        this.username = username;
        return this;
    }

    public DataBaseExport setPassword(String password) {
        this.password = password;
        return this;
    }

    public DataBaseExport setTables(String... tables) {
        this.tables = tables;
        return this;
    }

    public DataBaseExport setTableCreate(TableCreate tableCreate) {
        Assert.notNull(tableCreate, "tableCreate不能为空");
        this.tableCreate = tableCreate;
        return this;
    }

    public DataBaseExport setTableNameSensitive(boolean tableNameSensitive) {
        this.tableNameSensitive = tableNameSensitive;
        return this;
    }

    public DataBaseExport setColumnNameSensitive(boolean columnNameSensitive) {
        this.columnNameSensitive = columnNameSensitive;
        return this;
    }

    public String getJdbcUrl() {
        return jdbcUrl;
    }

    public String getDriverClassName() {
        return driverClassName;
    }

    public String getUsername() {
        return username;
    }

    public String getPassword() {
        return password;
    }

    public String[] getTables() {
        return tables;
    }

    public TableCreate getTableCreate() {
        return tableCreate;
    }

    public boolean isTableNameSensitive() {
        return tableNameSensitive;
    }

    public boolean isColumnNameSensitive() {
        return columnNameSensitive;
    }
}
