package com.elitescloud.boot.tenant.client;

import com.elitescloud.boot.condition.ConditionalOnTenant;
import com.elitescloud.boot.tenant.client.common.AbstractTenantDatasourceProvider;
import com.elitescloud.cloudt.common.constant.TenantIsolateStrategy;
import com.elitescloud.cloudt.context.util.database.DatasourceLoadUtil;
import com.zaxxer.hikari.HikariDataSource;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.boot.context.event.ApplicationContextInitializedEvent;
import org.springframework.context.ApplicationEvent;
import org.springframework.context.event.SmartApplicationListener;
import org.springframework.core.env.Environment;
import org.springframework.lang.NonNull;
import org.springframework.orm.jpa.vendor.Database;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import javax.sql.DataSource;

/**
 * 租户数据源监听配置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/4/4
 */
@ConditionalOnTenant(isolateStrategy = TenantIsolateStrategy.SCHEMA)
class CloudtTenantDataSourceListener implements SmartApplicationListener, InitializingBean {

    @Override
    public boolean supportsEventType(Class<? extends ApplicationEvent> eventType) {
        return eventType.isAssignableFrom(ApplicationContextInitializedEvent.class);
    }

    @Override
    public void onApplicationEvent(@NonNull ApplicationEvent event) {
        if (event instanceof ApplicationContextInitializedEvent) {
            Environment env = ((ApplicationContextInitializedEvent) event).getApplicationContext().getEnvironment();
            // 是否数据隔离
            TenantIsolateStrategy strategy = loadingDatabaseSeparate(env);
            if (strategy == null || strategy == TenantIsolateStrategy.NONE || strategy == TenantIsolateStrategy.FIELD) {
                return;
            }

            // 加载数据源
            DataSource dataSource = loadingDataSource(env);

            // 加载默认schema
            loadingDefaultSchema(env);

            // 加载数据库类型
            loadingDatabase(env, dataSource);
        }
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        Assert.hasText(AbstractTenantDatasourceProvider.getDefaultSchema(), "默认租户schema为空");
        Assert.notNull(AbstractTenantDatasourceProvider.getDefaultDataSource(), "默认数据源为空");
    }

    private TenantIsolateStrategy loadingDatabaseSeparate(Environment env) {
        String strategy = env.getProperty(TenantClientProperties.CONFIG_PREFIX + ".isolate-strategy");
        var isolateStrategy = TenantIsolateStrategy.parse(strategy);
        AbstractTenantDatasourceProvider.setIsolateStrategy(isolateStrategy);

        return isolateStrategy;
    }

    private Database loadingDatabase(Environment env, DataSource dataSource) {
        Database database = null;

        // 如果是hikari数据源
        if (dataSource instanceof HikariDataSource) {
            database = AbstractTenantDatasourceProvider.getDatabaseTypeByUrl(((HikariDataSource) dataSource).getJdbcUrl());
            if (database != null) {
                AbstractTenantDatasourceProvider.setDatabaseType(database);
                return database;
            }
        }

        // 从jpa指定的数据库类型
        String dbTypeStr = env.getProperty("spring.jpa.database");
        if (StringUtils.hasText(dbTypeStr)) {
            try {
                database = Database.valueOf(dbTypeStr.toUpperCase());
            } catch (IllegalArgumentException e) {
                e.printStackTrace();
            }
            if (database != null) {
                AbstractTenantDatasourceProvider.setDatabaseType(database);
                return database;
            }
        }

        return database;
    }

    private String loadingDefaultSchema(Environment env) {
        String schema = env.getProperty(TenantClientProperties.CONFIG_PREFIX + ".default-schema");
        if (StringUtils.hasText(schema)) {
            AbstractTenantDatasourceProvider.setDefaultSchema(schema);
        }
        return schema;
    }

    private DataSource loadingDataSource(Environment env) {
        DataSource dataSource = DatasourceLoadUtil.loadForHikari(env);

        AbstractTenantDatasourceProvider.setDefaultDataSource(dataSource);

        return dataSource;
    }
}
