package com.elitescloud.boot.common.annotation;

import com.elitescloud.boot.common.annotation.businessobject.*;

import java.lang.annotation.*;

/**
 * 业务对象操作.
 *
 * @author Kaiser（wang shao）
 * @date 2024/1/30
 */
@Documented
@Target(ElementType.METHOD)
@Retention(RetentionPolicy.RUNTIME)
@Inherited
public @interface BusinessObjectOperation {
    String BUSINESS_OBJECT_NAME_DEFAULT = "@BusinessObject";

    /**
     * 是否启用
     *
     * @return 默认是
     */
    boolean enabled() default true;

    /**
     * 操作类型
     *
     * @return 操作类型
     */
    OperationTypeEnum operationType();

    /**
     * 操作名称
     *
     * @return 操作名称
     */
    String operationDescription();

    /**
     * 详细描述
     *
     * @return 描述信息
     */
    String detail() default "-";

    /**
     * 业务对象
     * <p>
     * 业务操作的业务对象关联，如果不填写默认关联类上@BusinessObject内容
     *
     * @return 业务对象
     */
    String businessObjectType() default BUSINESS_OBJECT_NAME_DEFAULT;

    /**
     * 操作编码/空会自动生成
     *
     * @return 操作描述
     */
    String operationCode() default "";

    /**
     * 是否启用数据权限的支持
     *
     * @return 是否启用数据权限的支持
     */
    boolean dataPermissionEnabled() default false;

    /**
     * 是否自动过滤字段权限
     *
     * @return 是否自动过滤字段权限
     */
    boolean fieldPermissionAutoFilter() default false;

    /**
     * 权限规则同步关联的业务对象操作编码
     * <p>
     * 如在导出操作上关联了分页查询操作，则自动使用分页查询的权限规则配置
     *
     * @return 业务对象操作编码
     */
    String permissionRef() default "";

    /**
     * 业务参数
     * <p>
     * 用于业务查询
     *
     * @return 业务参数
     */
    String businessParam() default "";

    /**
     * 操作范围
     *
     * @return 操作范围
     */
    OperationScopeEnum operationScopeEnum() default OperationScopeEnum.PRIVATE_DOMAIN;

    /**
     * 操作状态
     *
     * @return 操作状态
     */
    OperationStateEnum operationStateEnum() default OperationStateEnum.RELEASED;

    /**
     * 是否记录日志
     *
     * @return 是否记录日志
     */
    boolean logEnabled() default false;

    /**
     * 是否记录入参
     *
     * @return 是否记录入参
     */
    boolean argsJsonEnabled() default false;

    /**
     * 是否记录出参
     *
     * @return 是否记录出参
     */
    boolean resultJsonEnabled() default false;

    /**
     * 业务入参对象的取值名称
     * <p>
     * 当前版本不支持
     *
     * @return 取值名称
     */
    BusinessParmObjectTypeEnum businessParamObjectType() default BusinessParmObjectTypeEnum.ALL;

    /**
     * 业务参数取值key
     * <p>
     * 可用于重复请求过滤的标识，如#p0.docNo可用来取值第一个参数对象里的docNo属性。优先级大于 {@link #businessParamValueKeyGenerator()}，例如：
     * <blockquote>
     *     <pre>
     *         {@code
     *             public class OrderVO implements Serializable {
     *                 private static final long serialVersionUID = -7507543161690997969L;
     *
     *                 private String docNo;
     *
     *                 // 省去get set方法
     *             }
     *
     *             @BusinessObjectOperation(allowRepeatRequest = false, businessParamValueKey = "#p0.docNo")
     *             public ApiResult<Long> saveOrder(@RequestBody OrderVO orderVO, @RequestParam("sn")) {
     *                 // 省去调用逻辑
     *             }
     *         }
     *     </pre>
     * </blockquote>
     *
     * @return 业务参数取值key
     */
    String businessParamValueKey() default "";

    /**
     * 业务参数取值key生成器
     * <p>
     * 必须是 {@link BusinessParamValueKeyGenerator}的实现类，且至少要有一个空的构造方法
     *
     * @return 实现类的class
     */
    Class<? extends BusinessParamValueKeyGenerator> businessParamValueKeyGenerator() default BusinessParamValueKeyGenerator.class;

    /**
     * 是否允许重复请求
     *
     * @return 默认允许重复请求
     */
    boolean allowRepeatRequest() default true;

    /**
     * 重复请求间隔时间
     * <p>
     * 单位毫秒，当值小于1时，使用系统的默认配置
     *
     * @return 间隔时间
     */
    int repeatIntervalMillis() default -1;
}
