package com.elitescloud.boot.common.servlet;

import cn.hutool.core.collection.CollUtil;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * 自定义Request上下文.
 * <p>
 * 用以解决请求结束后属性会被清空的问题。 {@link io.undertow.servlet.spec.HttpServletRequestImpl#freeResources()}
 *
 * @author Kaiser（wang shao）
 * @date 2023/12/28
 */
public class CloudtRequestAttributes {

    private final RequestAttributes requestAttributes;
    private final Map<String, Object> attributes;

    private CloudtRequestAttributes(RequestAttributes requestAttributes) {
        this.requestAttributes = requestAttributes;
        this.attributes = this.obtainAttributes(requestAttributes);
    }

    public static CloudtRequestAttributes getInstance() {
        var requestAttributes = RequestContextHolder.getRequestAttributes();
        if (requestAttributes == null) {
            return null;
        }

        return new CloudtRequestAttributes(requestAttributes);
    }

    public RequestAttributes getRequestAttributes() {
        if (requestAttributes == null) {
            return null;
        }

        // 设置属性
        if (CollUtil.isNotEmpty(attributes)) {
            var request = ((ServletRequestAttributes) requestAttributes).getRequest();
            for (Map.Entry<String, Object> entry : attributes.entrySet()) {
                request.setAttribute(entry.getKey(), entry.getValue());
            }

        }

        return requestAttributes;
    }

    private Map<String, Object> obtainAttributes(RequestAttributes requestAttributes) {
        if (requestAttributes == null) {
            return Collections.emptyMap();
        }

        var request = ((ServletRequestAttributes) requestAttributes).getRequest();
        Map<String, Object> attrMap = new HashMap<>(16);

        request.getAttributeNames().asIterator().forEachRemaining(attr -> attrMap.put(attr, request.getAttribute(attr)));

        return attrMap;
    }
}
