package com.elitescloud.boot.util;

import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.aop.aspectj.MethodInvocationProceedingJoinPoint;

import javax.validation.constraints.NotNull;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;

/**
 * AOP工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2024/5/10
 */
public class AspectUtil {

    private AspectUtil() {
    }

    /**
     * 获取目标类的对象
     *
     * @param joinPoint 连接点
     * @return 类对象
     */
    public static Object getTargetBean(@NotNull JoinPoint joinPoint) {
        return joinPoint.getTarget();
    }

    /**
     * 获取执行的方法
     *
     * @param joinPoint 连接点
     * @return 执行方法
     */
    public static Method getTargetMethod(@NotNull JoinPoint joinPoint) {
        var methodSignature = getMethodSignature(joinPoint);
        return methodSignature == null ? null : methodSignature.getMethod();
    }

    /**
     * 获取执行的方法
     *
     * @param joinPoint 连接点
     * @return 执行方法
     */
    public static MethodSignature getMethodSignature(@NotNull JoinPoint joinPoint) {
        if (!(joinPoint instanceof MethodInvocationProceedingJoinPoint)) {
            return null;
        }

        return (MethodSignature) joinPoint.getSignature();
    }

    /**
     * 获取注解
     * <p>
     * 优先获取方法，方法上没有再获取类上的
     *
     * @param point          连接点
     * @param annotationType 注解类型
     * @param <T>            注解类型
     * @return 注解
     */
    @SuppressWarnings("unchecked")
    public static <T extends Annotation> T[] getAnnotations(@NotNull JoinPoint point, @NotNull Class<T> annotationType) {
        var method = getTargetMethod(point);
        if (method != null) {
            var annotation = method.getAnnotationsByType(annotationType);
            if (ArrayUtil.isNotEmpty(annotation)) {
                return annotation;
            }
        }

        var targetBean = getTargetBean(point);
        return targetBean == null ? (T[]) new Annotation[0] : targetBean.getClass().getAnnotationsByType(annotationType);
    }

    /**
     * 获取注解
     * <p>
     * 优先获取方法，方法上没有再获取类上的
     *
     * @param point          连接点
     * @param annotationType 注解类型
     * @param <T>            注解类型
     * @return 注解
     */
    public static <T extends Annotation> T getAnnotation(@NotNull JoinPoint point, @NotNull Class<T> annotationType) {
        var annotations = getAnnotations(point, annotationType);
        return ArrayUtil.isEmpty(annotations) ? null : annotations[0];
    }
}
