package com.elitescloud.boot.util;

import cn.hutool.core.util.ClassUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.common.translate.TranslatorCustomizer;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import lombok.extern.slf4j.Slf4j;

import javax.validation.constraints.NotNull;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 数据翻译工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2024/5/9
 */
@Slf4j
public class DataTranslateUtil {

    /**
     * 翻译插件
     */
    private static List<TranslatorCustomizer> translatorCustomizers = null;
    private static TenantClientProvider tenantClientProvider = null;

    public static void translate(@NotNull Object data) {
        translate(data, false);
    }

    public static void translate(@NotNull Object data, boolean recursive) {
        translate(data, recursive, null);
    }

    public static void translate(@NotNull Object data, boolean recursive, Set<String> ignoredTranslators) {
        // 获取翻译器
        var translators = getTranslatorCustomizers();
        if (translators.isEmpty()) {
            return;
        }
        var tenant = getTenantClientProvider().getSessionTenant();

        // 拆包翻译
        ObjUtil.unwrap(data, obj -> {
            if (!supportTranslate(obj)) {
                // 不支持翻译
                return;
            }

            Set<String> executedTranslators = new HashSet<>(translators.size());
            for (TranslatorCustomizer translator : translators) {
                log.debug("开始使用{}进行翻译...", translator.code());

                if (ignoredTranslators != null && ignoredTranslators.contains(translator.code())) {
                    // 需要忽略的
                    continue;
                }
                if (executedTranslators.contains(translator.code())) {
                    // 已执行
                    continue;
                }
                executedTranslators.add(translator.code());

                // 开始翻译
                try {
                    executeTranslate(translator, obj, tenant, recursive);
                } catch (Exception e) {
                    log.error("翻译出现异常：{}", translator.code(), e);
                }
            }
        });
    }

    private static void executeTranslate(TranslatorCustomizer translator, Object data,
                                         SysTenantDTO tenant, boolean recursive) throws Exception {
        translator.translate(data, tenant);

        // 递归翻译
        if (!recursive || !translator.supportRecursive()) {
            return;
        }
        var fields = BeanWrapperUtil.getFields(data);
        Object value = null;
        for (var field : fields) {
            if (field.getReadMethod() == null) {
                continue;
            }
            if (!supportTranslate(field.getField().getType())) {
                // 不支持翻译
                continue;
            }

            value = field.getReadMethod().invoke(data);
            if (ObjUtil.isEmpty(value)) {
                // 数据为空
                continue;
            }

            executeTranslate(translator, value, tenant, recursive);
        }
    }

    private static boolean supportTranslate(Object data) {
        if (data instanceof Class) {
            // 根据类型判断，基本类型的不支持翻译
            return !ClassUtil.isSimpleValueType((Class<?>) data);
        }

        return !ObjUtil.isEmpty(data) && !ClassUtil.isSimpleValueType(data.getClass());
    }

    private static List<TranslatorCustomizer> getTranslatorCustomizers() {
        if (translatorCustomizers == null) {
            translatorCustomizers = SpringContextHolder.getObjectProvider(TranslatorCustomizer.class).stream()
                    .collect(Collectors.toList());
        }
        return translatorCustomizers;
    }

    private static TenantClientProvider getTenantClientProvider() {
        if (tenantClientProvider == null) {
            tenantClientProvider = SpringContextHolder.getBean(TenantClientProvider.class);
        }
        return tenantClientProvider;
    }
}
