package com.elitescloud.cloudt.context.util.database;

import com.zaxxer.hikari.HikariDataSource;
import lombok.extern.log4j.Log4j2;
import org.springframework.boot.context.properties.bind.BindResult;
import org.springframework.boot.context.properties.bind.Binder;
import org.springframework.core.env.Environment;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import javax.sql.DataSource;

/**
 * 数据源加载工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2022/5/16
 */
@Log4j2
public class DatasourceLoadUtil {

    /**
     * 加载hikari数据源
     *
     * @param environment 环境变量
     * @return 数据源
     */
    public static DataSource loadForHikari(Environment environment) {
        DataSource dataSource = loadingSharding(environment);

        if (dataSource == null) {
            dataSource = loadingHikari(environment);
        }

        return dataSource;
    }

    private static DataSource loadingSharding(Environment environment) {
        try {
            Class.forName("org.apache.shardingsphere.shardingjdbc.jdbc.core.datasource.ShardingDataSource");
        } catch (ClassNotFoundException e) {
            return null;
        }

        Boolean enabled = environment.getProperty("spring.shardingsphere.enabled", Boolean.class);
        if (Boolean.FALSE.equals(enabled)) {
            // 未启用
            return null;
        }

        String prefix = "spring.shardingsphere.datasource.";

        // 获取数据源名称，默认取第一个
        String datasourceName = environment.getProperty(prefix + "name");
        if (!StringUtils.hasText(datasourceName)) {
            datasourceName = environment.getProperty(prefix + "names");
            if (StringUtils.hasText(datasourceName)) {
                datasourceName = datasourceName.split(",")[0];
            }
        }
        if (!StringUtils.hasText(datasourceName)) {
            log.warn("未获取到有效的sharding数据源名称");
            return null;
        }

        // 初始化数据源
        String datasourceKey = prefix + datasourceName.trim();
        String datasourceType = environment.getProperty(datasourceKey + ".type");
        Assert.hasText(datasourceType, "未知shardingSphere配置的数据源类型");

        Class<DataSource> dataSourceClass = null;
        try {
            dataSourceClass = (Class<DataSource>) Class.forName(datasourceType);
        } catch (ClassNotFoundException e) {
            throw new RuntimeException("未找到数据源类型class：" + datasourceType, e);
        }

        BindResult<DataSource> dataSourceBindResult = Binder.get(environment).bind(datasourceKey, dataSourceClass);
        if (dataSourceBindResult.isBound()) {
            return dataSourceBindResult.get();
        }

        throw new RuntimeException("加载数据源失败");
    }

    private static HikariDataSource loadingHikari(Environment environment) {
        String prefixDatasource = "spring.datasource";
        String prefix = "spring.datasource.hikari";

        String type = environment.getProperty(prefixDatasource + ".type");
        if (!"com.zaxxer.hikari.HikariDataSource".equals(type)) {
            return null;
        }

        BindResult<HikariDataSource> dataSourceBindResult = Binder.get(environment).bind(prefix, HikariDataSource.class);
        if (dataSourceBindResult.isBound()) {
            HikariDataSource dataSource = dataSourceBindResult.get();

            if (!StringUtils.hasText(dataSource.getJdbcUrl())) {
                String url = environment.getProperty(prefixDatasource + ".url");
                if (StringUtils.hasText(url)) {
                    dataSource.setJdbcUrl(url);
                }
            }

            if (!StringUtils.hasText(dataSource.getUsername())) {
                String username = environment.getProperty(prefixDatasource + ".username");
                if (StringUtils.hasText(username)) {
                    dataSource.setUsername(username);
                }
            }

            if (!StringUtils.hasText(dataSource.getPassword())) {
                String password = environment.getProperty(prefixDatasource + ".password");
                if (StringUtils.hasText(password)) {
                    dataSource.setPassword(password);
                }
            }

            if (!StringUtils.hasText(dataSource.getDriverClassName())) {
                String driverClassName = environment.getProperty(prefixDatasource + ".driver-class-name");
                if (StringUtils.hasText(driverClassName)) {
                    dataSource.setDriverClassName(driverClassName);
                }
            }

            return dataSource;
        }

        return null;
    }
}
