package com.elitescloud.cloudt.platform.controller.api;

import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.CodeNameParam;
import com.elitescloud.cloudt.platform.model.params.udc.*;
import com.elitescloud.cloudt.platform.model.vo.SysPlatformUdcVO;
import com.elitescloud.cloudt.platform.service.SysPlatformUdcService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import java.util.Collections;
import java.util.List;

/**
 * @Description:
 * @author: niu.chen
 * @date: 2022.09.15
 **/
@Slf4j
@RestController
@Api(value = "平台udc", tags = {"平台udc"})
@RequestMapping("/sys/platform/udc")
public class SysPlatformUdcController {

    private final SysPlatformUdcService sysPlatformUdcService;

    public SysPlatformUdcController(SysPlatformUdcService sysPlatformUdcService) {
        this.sysPlatformUdcService = sysPlatformUdcService;
    }


    @PostMapping("/addUdc")
    @ApiOperation(value = "添加Udc", notes = "平台udc")
    public ApiResult<Long> addUdc(@RequestBody @Valid AddUdcParam addUdcParam) {
        return sysPlatformUdcService.addUdc(addUdcParam);
    }
    @PostMapping("/addUdcValue")
    @ApiOperation(value = "添加addUdcValue", notes = "平台udc")
    public ApiResult<Long> addUdcValue(@RequestBody @Valid AddUdcValueParam addUdcParam) {
        return sysPlatformUdcService.addUdcValue(addUdcParam);
    }

    /**
     * UDC值编辑
     *
     * @param saveVO UDC值信息
     * @return 值ID
     */
    @ApiOperation(value = "UDC值编辑")
    @PostMapping(value = "/updateUdcValue")
    public ApiResult<Long> updateUdcValue(@RequestBody @Valid UdcValueSaveVO saveVO) {
        return sysPlatformUdcService.updateUdcValue(saveVO);
    }

    @ApiOperation(value = "删除Udc", notes = "平台udc")
    @PutMapping("/deleteFlagUdc/{id}")
    public ApiResult<Boolean> deleteFlagUdc(@PathVariable Long id) {

        return sysPlatformUdcService.deleteUdc(id);
    }

    @ApiOperation(value = "删除deleteFlagUdcValue", notes = "平台udc")
    @PutMapping("/deleteFlagUdcValue/{id}")
    public ApiResult<Boolean> deleteFlagUdcValue(@PathVariable Long id) {
        return sysPlatformUdcService.deleteUdcValue(id);
    }


    @PutMapping("/updateUdc/{id}")
    @ApiOperation(value = "更新Udc", notes = "平台udc")
    public ApiResult<Boolean> updateUdc(@PathVariable Long id, @RequestBody @Valid UpdateUdcParam updateUdcParam) {
        return sysPlatformUdcService.updateUdc(id, updateUdcParam);
    }

    @PutMapping("/updateUdcValueAllowStart")
    @ApiOperation(value = "更新UdcValue是否启用", notes = "平台udc")
    public ApiResult<Boolean> updateUdcValueAllowStart(@RequestBody @Valid UpdateUdcValueAllowStartParam updateUdcValueAllowStart) {
        return sysPlatformUdcService.updateUdcValueAllowStart(updateUdcValueAllowStart.getId(),
                updateUdcValueAllowStart.getAllowStart());
    }


    @GetMapping("/getUdc/{id}")
    @ApiOperation(value = "id查询UDC和value", notes = "平台udc")
    public ApiResult<SysPlatformUdcVO> getUdc(@PathVariable Long id) {
        return sysPlatformUdcService.getUdc(id);
    }

    @PostMapping("/queryUdc")
    @ApiOperation(value = "分页查询UDC", notes = "平台udc")
    public ApiResult<PagingVO<SysPlatformUdcVO>> queryUdc( @RequestBody @Valid QueryUdcParam queryUdcParam) {
        return sysPlatformUdcService.queryUdc(queryUdcParam);
    }

    /**
     * 根据udc编码查询UDC值列表
     *
     * @param appCode 应用编码
     * @param udcCode UDC编码
     * @return udc值列表
     */
    @ApiOperation(value = "根据udc编码查询UDC值列表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "appCode", value = "应用编码", required = true),
            @ApiImplicitParam(name = "udcCode", value = "UDC编码", required = true),
    })
    @GetMapping(value = "/valueList")
    public ApiResult<List<CodeNameParam>> udcValues(@NotBlank(message = "应用编码为空") @RequestParam(name = "appCode") String appCode,
                                                    @NotBlank(message = "UDC编码为空")  @RequestParam(name = "udcCode") String udcCode) {
        return sysPlatformUdcService.getValueList(appCode, udcCode);
    }
}
