package com.elitescloud.cloudt.platform.service.impl;

import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.CodeNameParam;
import com.elitescloud.cloudt.common.constant.CacheKey;
import com.elitescloud.cloudt.common.exception.BusinessException;
import com.elitescloud.cloudt.common.util.RedisUtils;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.platform.convert.AppConvert;
import com.elitescloud.cloudt.platform.model.constant.PlatformAppTypeEnum;
import com.elitescloud.cloudt.platform.model.entity.QSysPlatformAppDO;
import com.elitescloud.cloudt.platform.model.params.app.AddAppParam;
import com.elitescloud.cloudt.platform.model.params.app.QueryAppParam;
import com.elitescloud.cloudt.platform.model.params.app.UpdateAppParam;
import com.elitescloud.cloudt.platform.model.vo.SysPlatformAppVO;
import com.elitescloud.cloudt.platform.service.SysPlatformAppService;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformAppRepo;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformAppRepoProc;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformMenusRepo;
import com.elitescloud.cloudt.system.vo.SysUdcVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.dsl.Expressions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

/**
 * @Description:
 * @author: niu.chen
 * @date: 2022.09.15
 **/
@Service
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
@Slf4j
@RequiredArgsConstructor
public class SysPlatformAppServiceImpl implements SysPlatformAppService {

    private final SysPlatformAppRepo sysPlatformAppRepo;
    private final SysPlatformAppRepoProc sysPlatformAppRepoProc;
    private final RedisUtils redisUtils;
    private final SysPlatformMenusRepo sysPlatformMenusRepo;

    @Override
    public ApiResult<Long> addApp(AddAppParam addAppParam) {

        var list = sysPlatformAppRepo.findByAppCode(addAppParam.getAppCode());
        if (!list.isEmpty()) {
            return ApiResult.fail("AppCode重复");
        }
        var saveDo = AppConvert.INSTANCE.saveParamToDo(addAppParam);
        saveDo.setAppState(true);
        var id = sysPlatformAppRepo.save(saveDo).getId();

        // 清空缓存
        clearCache();

        return ApiResult.ok(id);
    }

    @Override
    public ApiResult<Boolean> deleteFlagApp(Long id) {
        sysPlatformAppRepo.findById(id).ifPresentOrElse(sysPlatformAppDO ->
                {
                    sysPlatformAppDO.setDeleteFlag(1);
                    sysPlatformAppRepo.save(sysPlatformAppDO);

                    // 清空缓存
                    clearCache();
                }
                ,
                () -> {
                    throw new BusinessException("id不存在");
                });

        return ApiResult.ok(true);
    }

    @Override
    public ApiResult<Boolean> deleteApp(Long id) {
        var appDo = sysPlatformAppRepo.findById(id);
        if (appDo.isPresent()) {
            var menusList = sysPlatformMenusRepo.findAllByMenusAppCode(appDo.get().getAppCode());
            if (menusList != null && menusList.size() > 0) {
                return ApiResult.fail("请先删除APP下的菜单");
            }
            log.info("App应用删除：{}", appDo.get());
            sysPlatformAppRepo.deleteById(id);
            // 清空缓存
            clearCache();
        } else {
            throw new BusinessException("id不存在");
        }
        return ApiResult.ok(true);
    }

    @Override
    public ApiResult<Boolean> updateApp(Long id, UpdateAppParam updateAppParam) {
        AtomicReference<Boolean> allowUpdate = new AtomicReference<>(false);
        sysPlatformAppRepo.findById(id).ifPresentOrElse(sysPlatformAppDO -> {
//                    if (!sysPlatformAppDO.getAppCode().equals(updateAppParam.getAppCode())) {
//                        var list = sysPlatformAppRepo.findByAppCode(updateAppParam.getAppCode());
//                        if (!list.isEmpty()) {
//                            allowUpdate.set(false);
//                            return;
//                            // throw new BusinessException("Code已经存在重复:" + updateAppParam.getAppCode());
//                        }
//                    }
//                    sysPlatformAppDO.setAppCode(updateAppParam.getAppCode());
                    sysPlatformAppDO.setAppName(updateAppParam.getAppName());
                    sysPlatformAppDO.setAppType(updateAppParam.getAppType());
                    sysPlatformAppDO.setAppDescribe(updateAppParam.getAppDescribe());
                    sysPlatformAppDO.setAppOrder(updateAppParam.getAppOrder());
                    sysPlatformAppRepo.save(sysPlatformAppDO);
                    allowUpdate.set(true);

                    // 清空缓存
                    clearCache();
                },
                () -> {
                    throw new BusinessException("id不存在");
                });
        if (allowUpdate.get()) {
            return ApiResult.ok(true);
        } else {
            return ApiResult.fail("Code已经存在,重复:");
        }
    }

    @Override
    public ApiResult<Boolean> updateAppState(Long id, Boolean state) {
        sysPlatformAppRepo.findById(id).ifPresentOrElse(sysPlatformAppDO ->
                {
                    sysPlatformAppDO.setAppState(state);
                    sysPlatformAppRepo.save(sysPlatformAppDO);

                    // 清空缓存
                    clearCache();
                },

                () -> {
                    throw new BusinessException("id不存在");
                });
        return ApiResult.ok();
    }

    @Override
    public ApiResult<SysPlatformAppVO> getApp(Long id) {
        var appDO = sysPlatformAppRepo.findById(id);
        if (appDO.isPresent()) {
            return ApiResult.ok(AppConvert.INSTANCE.sysPlatformAppDOToSysPlatformAppVO(appDO.get()));
        } else {
            return ApiResult.fail("id不存在");
        }
    }

    @Override
    public ApiResult<PagingVO<SysPlatformAppVO>> pageMng(QueryAppParam queryAppParam) {
        var pagingVo
                = sysPlatformAppRepoProc.pageMng(queryAppParam)
                .map(AppConvert.INSTANCE::sysPlatformAppDOToSysPlatformAppVO);
        return ApiResult.ok(pagingVo);
    }

    @Override
    public ApiResult<List<SysUdcVO>> getAppType() {

        return ApiResult.ok(
                List.of(PlatformAppTypeEnum.APP_TYPE_SYS.getUdcVO()
                        , PlatformAppTypeEnum.APP_TYPE_BUS.getUdcVO()
                )
        );
    }

    @Override
    public ApiResult<List<CodeNameParam>> all() {
        var result = sysPlatformAppRepoProc.allEnabled();
        try {
            redisUtils.set(CacheKey.CACHE_APP_ALL, result);
        } catch (Exception e) {
            log.error("缓存APP异常", e);
        }

        return ApiResult.ok(result);
    }

    @Override
    public ApiResult<PagingVO<SysPlatformAppVO>> queryByPage(QueryAppParam queryAppParam) {
        var pagingVo
                = sysPlatformAppRepoProc.pageQuery(queryAppParam)
                .map(AppConvert.INSTANCE::sysPlatformAppDOToSysPlatformAppVO);
        return ApiResult.ok(pagingVo);
    }

    private void clearCache() {
        try {
            redisUtils.del(CacheKey.CACHE_APP_ALL);
        } catch (Exception e) {
            log.error("清空app缓存异常", e);
        }
    }
}
