package com.elitescloud.cloudt.lowcode.dynamic.service.dynamic;

import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.context.AnalysisContext;
import com.alibaba.excel.event.AnalysisEventListener;
import com.alibaba.excel.metadata.CellData;
import com.elitescloud.boot.excel.util.ExcelUtil;
import com.elitescloud.cloudt.lowcode.dynamic.model.DynamicFieldCondition;
import com.elitescloud.cloudt.lowcode.dynamic.model.entity.DynamicBoFieldDefinitionDO;
import com.elitescloud.cloudt.lowcode.dynamic.model.param.DynamicFieldQueryParam;
import lombok.extern.slf4j.Slf4j;
import org.jooq.Condition;
import org.jooq.DataType;
import org.jooq.Field;
import org.jooq.impl.DSL;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.net.URLEncoder;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;


/**
 * @author : chen.niu
 * @description :
 * @date : 2024-03-27 22:28
 */
@Slf4j
public class DynamicUtil {
    public static LocalDateTime getOperationDateTime(String operationTime) {
        LocalDateTime  operationDateTime=null;
        if(operationTime!=null){
            if(operationTime.length()>19){
                operationTime=operationTime.substring(0,19);
            }
            DateTimeFormatter pattern=DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss");
            try{
                operationDateTime=LocalDateTime.parse(operationTime,pattern);
            }catch (Exception e){
                e.printStackTrace();
            }
        }
        return operationDateTime;
    }
    private static Object getOperationDate(String operationTime) {
        LocalDateTime  operationDateTime=null;
        if(operationTime!=null){
            DateTimeFormatter pattern=DateTimeFormatter.ofPattern("yyyy-MM-dd");
            try{
                operationDateTime=LocalDateTime.parse(operationTime,pattern);
            }catch (Exception e){
                e.printStackTrace();
            }
        }
        return operationDateTime;
    }
    public static Object transformValue(Object value, DynamicBoFieldDefinitionDO definitionDo) {
        try {
            if(value==null){
                return null;
            }
            if (definitionDo.getBasicType().getJooqDataType().isNumeric()) {
                return Double.parseDouble(value.toString());
            }else if (definitionDo.getBasicType().getJooqDataType().isDateTime()) {
                return  getOperationDateTime(value.toString());
            }else if (definitionDo.getBasicType().getJooqDataType().isDate()) {
                return  getOperationDate(value.toString());
            }
        } catch (NumberFormatException e) {
            String errMsg = definitionDo.getBasicName() + "数值转换错误" + value;
            log.error(errMsg, e);
            throw new RuntimeException(errMsg);
        }
        return value;
    }



    public static Field<?> getJooqField(DynamicBoFieldDefinitionDO field) {
        if (field == null) {
            return null;
        }
        DataType<?> dataType = field.getBasicType().getJooqDataType();
        if (field.getDbFieldLength() != null && dataType.hasLength()) {
            dataType = dataType.length(field.getDbFieldLength());
        }
        if (field.getDbFieldPrecision() != null && dataType.hasPrecision()) {
            dataType = dataType.precision(field.getDbFieldPrecision(), field.getDbFieldScale());
        }
        if (field.getDbFieldNullable() != null) {
            dataType = dataType.nullable(field.getDbFieldNullable());
        }
        Field<?> jooqField = DSL.field(DSL.name(field.getBasicKey()), dataType);
        return jooqField;
    }

    public static Condition getDynamicFieldQueryParamCondition(DynamicFieldQueryParam queryParam) {
        Condition condition = DSL.trueCondition(); // 初始化为真条件
        if (queryParam.getDynamicFieldConditionList() == null) {
            return condition;
        }
        for (DynamicFieldCondition fieldCondition : queryParam.getDynamicFieldConditionList()) {
            Field<Object> field = DSL.field(DSL.name(fieldCondition.getField()), Object.class);
            switch (fieldCondition.getCondition()) {
                case Equal:
                    condition = condition.and(field.eq(fieldCondition.getValue()));
                    break;
                case NotEqual:
                    condition = condition.and(field.ne(fieldCondition.getValue()));
                    break;
                case GreaterEqual:
                    condition = condition.and(field.ge(fieldCondition.getValue()));
                    break;
                case GreaterThan:
                    condition = condition.and(field.gt(fieldCondition.getValue()));
                    break;
                case LessEqual:
                    condition = condition.and(field.le(fieldCondition.getValue()));
                    break;
                case LessThan:
                    condition = condition.and(field.lt(fieldCondition.getValue()));
                    break;
                case Contain:
                    condition = condition.and(field.like("%" + fieldCondition.getValue() + "%"));
                    break;
                case StartWith:
                    condition = condition.and(field.like(fieldCondition.getValue() + "%"));
                    break;
                case EndWith:
                    condition = condition.and(field.like("%" + fieldCondition.getValue()));
                    break;
                case Between:
                    // 假设 fieldCondition.getValue() 是一个包含两个元素的数组或列表，表示范围
                    String[] range = fieldCondition.getValue().split(";");
                    condition = condition.and(field.between(range[0], range[1]));
                    break;
                case NotBetween:
                    // 同样假设 fieldCondition.getValue() 是一个包含两个元素的数组或列表
                    range = fieldCondition.getValue().split(";");
                    condition = condition.and(field.notBetween(range[0], range[1]));
                    break;
                case InList:
                    // 假设 fieldCondition.getValue() 是一个列表或数组，表示可能的值
                    condition = condition.and(field.in(fieldCondition.getValue()));
                    break;
                case NotIn:
                    // 同样假设 fieldCondition.getValue() 是一个列表或数组
                    condition = condition.and(field.notIn(fieldCondition.getValue()));
                    break;
                case IsNull:
                    condition = condition.and(field.isNull());
                    break;
                case NotNull:
                    condition = condition.and(field.isNotNull());
                    break;
                case Empty:
                    // 对于字符串字段，可以使用 LIKE 语句检查空字符串
                    condition = condition.and(field.like(""));
                    break;
                case NotEmpty:
                    // 对于字符串字段，可以使用 NOT LIKE 语句检查非空字符串
                    condition = condition.and(field.notLike(""));
                    break;
                default:
                    throw new IllegalArgumentException("Unsupported condition: " + fieldCondition.getCondition());
            }
        }
        return condition;
    }


    public static void writeResponseXlsx(String tableName, HttpServletResponse response, List<List<String>> head, List<List<Object>> dataList) throws IOException {
        String currentTime = LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyyMMddHHmmssSSS")); // 格式化当前时间为字符串
        // 设置响应头
        String encodedFileName = URLEncoder.encode(tableName, "UTF-8");
        response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        response.setCharacterEncoding("UTF-8");
        response.setHeader("Content-Disposition", "attachment; filename=\"" + URLEncoder.encode(encodedFileName + currentTime) + ".xlsx\"");

        // 使用EasyExcel导出数据
        ExcelUtil.normalizeExcelWriter(EasyExcel.write(response.getOutputStream())
                        .head(head))
                .sheet("Sheet1")
                .doWrite(dataList);
    }


    /**
     * 读取Excel文件并将每一行转换成一个Map，其中key是列的表头（字段名），value是单元格的值。
     * 使用LinkedHashMap保证插入和读取的顺序一致。
     *
     * @param file MultipartFile 上传的Excel文件
     * @return List<Map < String, String>> 每个Map代表一行数据，键为表头，值为对应的单元格数据
     * @throws IOException 如果读取Excel文件出错
     */
    public static List<Map<String, Object>> readExcelAsMap(MultipartFile file) throws IOException {
        List<Map<String, Object>> list = new ArrayList<>();
        EasyExcel.read(file.getInputStream(), new AnalysisEventListener<Map<Integer, String>>() {
            // 使用一个标志来标记当前是否是第一行（即表头）
            private boolean headFlag = true;
            private List<String> headList = new ArrayList<>();
            private int rowIndex = 0;

            @Override
            public void invoke(Map<Integer, String> data, AnalysisContext context) {
                // 跳过第二行
                if (rowIndex > 1) { // rowIndex > 1，意味着从第三行开始处理数据
                    Map<String, Object> rowMap = new LinkedHashMap<>();
                    data.forEach((key, value) -> {
                        // 防止索引越界，检查headList大小
                        if (key < headList.size()) {
                            rowMap.put(headList.get(key), value);
                        }
                    });
                    list.add(rowMap);
                }
                rowIndex++;

            }

            @Override
            public void doAfterAllAnalysed(AnalysisContext context) {
                // 所有数据解析完成后调用
            }

            @Override
            public void invokeHead(Map<Integer, CellData> headMap, AnalysisContext context) {
                // 仅处理第一行，获取字段名
                if (rowIndex == 0) {
                    headMap.forEach((key, value) -> headList.add(value.getStringValue()));
                }
                rowIndex++;
            }
        }).sheet().doRead();
        return list;
    }


}
