package com.elitescloud.cloudt.lowcode.dynamic.service.dynamic.impl;

import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.lowcode.dynamic.model.entity.DynamicBoFieldDefinitionDO;
import com.elitescloud.cloudt.lowcode.dynamic.model.param.BoFieldQueryParam;
import com.elitescloud.cloudt.lowcode.dynamic.model.param.DynamicExportParam;
import com.elitescloud.cloudt.lowcode.dynamic.model.param.DynamicFieldQueryParam;
import com.elitescloud.cloudt.lowcode.dynamic.model.vo.DynamicImportFailures;
import com.elitescloud.cloudt.lowcode.dynamic.model.vo.DynamicImportResult;
import com.elitescloud.cloudt.lowcode.dynamic.service.db.DbAuditFieldService;
import com.elitescloud.cloudt.lowcode.dynamic.service.db.DbTableService;
import com.elitescloud.cloudt.lowcode.dynamic.service.dynamic.DynamicBoApiService;
import com.elitescloud.cloudt.lowcode.dynamic.service.dynamic.DynamicBoModelService;
import com.elitescloud.cloudt.lowcode.dynamic.service.dynamic.DynamicUtil;
import lombok.extern.slf4j.Slf4j;
import org.jooq.*;
import org.jooq.impl.DSL;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author : chen.niu
 * @description :
 * @date : 2024-04-02 19:36
 */
@Service
@Slf4j
public class DynamicBoApiServiceImpl implements DynamicBoApiService {
    public static final String TENANT_ID = "tenant_id";
    public static final String ID = "id";
    //    @Autowired
//    private ApplicationContext applicationContext;


   private final DSLContext dsl;

   private final DbTableService dbTableService;


   private final DynamicBoFieldDefinitionServiceImpl boFieldDefinitionService;

   private final DbAuditFieldService dbAuditFieldService;

   private final DynamicBoModelService boModelService;

    public DynamicBoApiServiceImpl(DSLContext dsl, DbTableService dbTableService, DynamicBoFieldDefinitionServiceImpl boFieldDefinitionService, DbAuditFieldService dbAuditFieldService, DynamicBoModelService boModelService) {
        this.dsl = dsl;
        this.dbTableService = dbTableService;
        this.boFieldDefinitionService = boFieldDefinitionService;
        this.dbAuditFieldService = dbAuditFieldService;
        this.boModelService = boModelService;
    }

    @Override
    public ApiResult<Object> insert(String boModelCode,
                                    Map<String, Object> entityData) {
        try {
            var boModelDo = boModelService.getBoModelVoByCode(boModelCode);
            String tableName = boModelDo.getDatabaseTableName();
            List<DynamicBoFieldDefinitionDO> boFieldDefinitionDoList = boModelDo.getDynamicBoFieldDefinitionDoList();
            Map<String, DynamicBoFieldDefinitionDO> fieldDefinitionMap = buildFieldDefinitionMap(boFieldDefinitionDoList);

            boFiledCheck(fieldDefinitionMap, entityData);

            Table<?> table = DSL.table(DSL.name(tableName));
            List<Field<?>> fields = new ArrayList<>();
            List<Object> values = new ArrayList<>();

            dbAuditFieldService.fillInsertAuditFields(entityData);
            entityData.forEach((key, value) -> {
                DynamicBoFieldDefinitionDO dynamicBoFieldDefinitionDO = fieldDefinitionMap.get(key);
                if (dynamicBoFieldDefinitionDO != null) {
                    org.jooq.Field<?> jooqField = DynamicUtil.getJooqField(dynamicBoFieldDefinitionDO);
                    fields.add(jooqField);
                    values.add(DynamicUtil.transformValue(value, dynamicBoFieldDefinitionDO));
                } else {
                    org.jooq.Field<?> jooqField = DSL.field(DSL.name(key));
                    fields.add(jooqField);
                    values.add(value);
                }
            });

            InsertValuesStepN<?> insertStep = dsl
                    .insertInto(table)
                    .columns(fields)
                    .values(values);

            // 执行插入
            insertStep.execute();

            return ApiResult.ok(entityData.get("id"));
        } catch (Exception e) {
            log.error("插入数据时发生错误", e);
            // 根据你的错误处理策略返回一个错误结果
            return ApiResult.fail("插入数据失败");
        }
    }





    @Override
    public ApiResult<Integer> update(String boModelCode, Long id,
                                     Map<String, Object> entityData) {
        if(boModelCode == null || id == null || entityData == null || entityData.isEmpty()) {
            // 对输入参数进行基本的非空和边界条件校验
            return ApiResult.fail("Invalid input parameters");
        }

        var boModelDo = boModelService.getBoModelVoByCode(boModelCode);
        if(boModelDo == null) {
            return ApiResult.fail("BO Model not found for code: " + boModelCode);
        }

        String tableName = boModelDo.getDatabaseTableName();
        List<DynamicBoFieldDefinitionDO> boFieldDefinitionDoList = getBoFieldDefinitionByBoModelCodeDos(boModelCode);
        if(boFieldDefinitionDoList == null || boFieldDefinitionDoList.isEmpty()) {
            return ApiResult.fail("Field definitions not found for BO Model code: " + boModelCode);
        }
        // 构建从字段名到DynamicBoFieldDefinitionDo的映射，优化性能
        Map<String, DynamicBoFieldDefinitionDO> fieldDefinitionMap =buildFieldDefinitionMap(boFieldDefinitionDoList);

        boFiledCheck(fieldDefinitionMap, entityData);
        dbAuditFieldService.fillUpdateAuditFields(entityData);

        Table<?> table = DSL.table(DSL.name(tableName));
        UpdateQuery<?> updateQuery = dsl.updateQuery(table);



        // 动态添加更新字段和值
        entityData.forEach((key, value) -> {

                DynamicBoFieldDefinitionDO dynamicBoFieldDefinitionDO = fieldDefinitionMap.get(key);
                if(dynamicBoFieldDefinitionDO != null){
                    org.jooq.Field jooqField = DynamicUtil.getJooqField(dynamicBoFieldDefinitionDO);
                    updateQuery.addValue(jooqField, DynamicUtil.transformValue(value, dynamicBoFieldDefinitionDO));
                } else {
                    Field<Object> field = DSL.field(DSL.name(key), Object.class);
                    updateQuery.addValue(field, value);
                }


        });

        // 应用WHERE条件
        Field<Long> idField = DSL.field(DSL.name("id"), Long.class);
        Condition condition = idField.eq(id);
        condition = setTenantIdCondition(condition);
        updateQuery.addConditions(condition);

        // 执行更新操作
        try {
            int rowsUpdated = updateQuery.execute();
            return ApiResult.ok(rowsUpdated);
        } catch (Exception e) {
            // 处理可能的数据库异常
            return ApiResult.fail("Database update failed: " + e.getMessage());
        }
    }
    @Override
    @Transactional
    public ApiResult<Integer> updateBatch(String boModelCode, Map<Long,Map<String, Object>> entityDataMap) {
        entityDataMap.forEach((key,value)->{
            update(boModelCode,key,value);
        });
        return ApiResult.ok(entityDataMap.size());
    }

    @Override
    public ApiResult<Page<?>> queryPage(String boModelCode,
                                        DynamicFieldQueryParam queryParam) {
        var boModelDo = boModelService.getBoModelVoByCode(boModelCode);
        String tableName = boModelDo.getDatabaseTableName();
        Table<?> table = DSL.table(DSL.name(tableName));
        // 构建基本查询
        SelectJoinStep<?> baseQuery = dsl.select().from(table);
//        SelectConditionStep<?> query = dsl
//                .select() // 这里可以具体指定需要选择的字段
//                .from(table)
//                .join(DSL.table(DSL.name("other_table"))) // 示例：添加一个JOIN到其他表
//                .on(DSL.field(DSL.name("table.field")).eq(DSL.field(DSL.name("other_table.other_field")))) // 指定JOIN条件
//                .where(DSL.field(DSL.name("table.condition_field")).eq("someValue")); // 添加WHERE条件

        // 构造条件查询
        Condition condition = DynamicUtil.getDynamicFieldQueryParamCondition(queryParam);
        //租户条件
        condition = setTenantIdCondition(condition);
        SelectConditionStep<?> conditionalQuery = baseQuery.where(condition);
        // 应用排序
        var sortFields = queryParam.getOrders().stream()
                .map(order -> {
                    Field<Object> orderField = DSL.field(DSL.name(order.getColumn()), Object.class);
                    return order.isAsc() ? orderField.asc() : orderField.desc();
                })
                .toArray(SortField[]::new);
        conditionalQuery.orderBy(sortFields);

        // 应用分页并获取结果
        int offset = queryParam.getCurrent() * queryParam.getSize();
        Result<?> result = conditionalQuery
                .limit(queryParam.getSize())
                .offset(offset)
                .fetch();

        // 计算总数，重用相同的条件
        int total = dsl.selectCount().from(table).where(condition).fetchOne(0, int.class);

        // 将JOOQ结果转换为Page对象
        Page<?> pageResult = new PageImpl<>(result.intoMaps(),
                PageRequest.of(queryParam.getCurrent(), queryParam.getSize()), total);


        return ApiResult.ok(pageResult);
    }


    @Override
    public ApiResult<Map> getById(String boModelCode,
                                  Long id) {
        try {
            var boModelDo = boModelService.getBoModelVoByCode(boModelCode);
            String tableName = boModelDo.getDatabaseTableName();
            Table<?> table = DSL.table(DSL.name(tableName));
//            // 获取表对应的Table对象
//            if (table == null) {
//                // 如果表不存在，返回错误信息
//                return ApiResult.fail("Table not found: " + tableName);
//            }
            // 创建查询条件
            Condition condition = DSL.field(DSL.name(ID)).eq(id);
            //设置租户条件
            condition = setTenantIdCondition(condition);
            // 执行查询
            org.jooq.Record record = dsl.selectFrom(table).where(condition).fetchOne();
            if (record == null) {
                return ApiResult.fail("未找到数据");
            }
            Map<String, Object> map = record.intoMap();
            return ApiResult.ok(map);
        } catch (Exception e) {
            // 处理异常
            log.error("动态查询异常：", e);
            return ApiResult.fail("动态查询异常 data by ID: " + e.getMessage());
        }
    }

    @Override
    public void dynamicExport(String boModelCode, DynamicExportParam dynamicExportParam, HttpServletResponse response) throws IOException {
        if (dynamicExportParam == null || dynamicExportParam.getFieldMap() == null) {
            throw new RuntimeException("参数不能为空");
        }
        var boModelDo = boModelService.getBoModelVoByCode(boModelCode);
        String tableName = boModelDo.getDatabaseTableName();
        // 动态构建双层表头
        List<List<String>> head = new ArrayList<>();
        dynamicExportParam.getFieldMap().forEach((englishName, chineseName) -> {
            List<String> headerColumn = new ArrayList<>();
            headerColumn.add(englishName); // 第一行英文
            headerColumn.add(chineseName); // 第二行中文
            head.add(headerColumn);
        });

        // 构建查询字段
        List<Field<?>> selectFields = dynamicExportParam.getFieldMap().keySet().stream()
                .map(DSL::field)
                .collect(Collectors.toList());
        Table<?> table = DSL.table(DSL.name(tableName));

        var condition = DynamicUtil.getDynamicFieldQueryParamCondition(dynamicExportParam);
        //设置租户条件
        condition = setTenantIdCondition(condition);
        //查询执行
        Result<?> result = dsl.select(selectFields)
                .from(table)
                .where(condition)
                .orderBy(dynamicExportParam.getOrders().stream()
                        .map(order -> DSL.field(DSL.name(order.getColumn()), Object.class)
                                .sort(order.isAsc() ? SortOrder.ASC : SortOrder.DESC))
                        .toArray(SortField[]::new)).fetch();
        List<List<Object>> dataList = new ArrayList<>();
        if (result != null && result.size() > 0) {
            // 转换数据为 List<List<String>>，其中内部列表代表一行数据
            result.intoMaps().stream().map(map -> map.entrySet().stream()
                            .sorted(Map.Entry.comparingByKey()) // 可能需要根据实际需要对字段排序，以确保顺序与表头对应
                            .map(stringObjectEntry -> {
                                if (Timestamp.class.isAssignableFrom(stringObjectEntry.getClass())) {
                                    return ((Timestamp) stringObjectEntry.getValue()).toLocalDateTime();
                                }
                                return stringObjectEntry.getValue();
                            })
                            .collect(Collectors.toList()))
                    .collect(Collectors.toList());
        }

        try {
            DynamicUtil.writeResponseXlsx(tableName, response, head, dataList);
        } catch (Exception e) {
            log.error("导出Excel失败", e);
            throw e;
        }
    }


    @Override
    public ApiResult<DynamicImportResult> dynamicImport(String boModelCode,
                                                        MultipartFile file) {
        var boModelDo = boModelService.getBoModelVoByCode(boModelCode);
        String tableName = boModelDo.getDatabaseTableName();
        DynamicImportResult dynamicImportResult = new DynamicImportResult();
        List<Map<String, Object>> dataList = null;
        try {
            //读取excel数据
            dataList = DynamicUtil.readExcelAsMap(file);

        } catch (Exception e) {
            return ApiResult.fail("文件读取失败" + e.getMessage());
        }


        // 获取数据库表对象
        // Table<?> table = dsl.table(tableName);
        Table table = DSL.table(DSL.name(tableName));
        int row = 0;
        // 动态构建插入语句
        for (Map<String, Object> rowData : dataList) {
            row++;
            try {
                InsertSetStep<org.jooq.Record> insertStep = dsl.insertInto(table);
                dbAuditFieldService.fillInsertAuditFields(rowData);
                rowData.forEach((key, value) -> {
                    insertStep.set(DSL.field(key), value);
                });
                insertStep.columns().execute();
            } catch (Exception e) {
                DynamicImportFailures dynamicImportFailures = new DynamicImportFailures();
                dynamicImportFailures.setFailuresDate(rowData);
                dynamicImportFailures.setFailuresDescription("執行异常：" + e.getMessage());
                dynamicImportFailures.setFailuresRow(row);

                dynamicImportResult.getFailuresDescription().add(dynamicImportFailures);
                dynamicImportResult.setFailures(dynamicImportResult.getFailures() + 1);
                continue;
            }
            dynamicImportResult.setSuccessful(dynamicImportResult.getSuccessful() + 1);
        }

        return ApiResult.ok(dynamicImportResult);

    }

    @Override
    public ApiResult<?> delete(String boModelCode, Long id) {
        var boModelDo = boModelService.getBoModelVoByCode(boModelCode);
        String tableName = boModelDo.getDatabaseTableName();
        List<DynamicBoFieldDefinitionDO> boFieldDefinitionDoList = getBoFieldDefinitionByBoModelCodeDos(boModelCode);
        if (boFieldDefinitionDoList == null || boFieldDefinitionDoList.isEmpty()) {
            throw new BusinessException("未找到Bo字段");
        }
        var condition = DSL.field(DSL.name(ID)).eq(id);
        //设置租户条件
        setTenantIdCondition(condition);
        Table<?> table = DSL.table(DSL.name(tableName));
        dsl.deleteFrom(table)
                .where(condition)
                .execute();
        return ApiResult.ok();
    }

    @Override
    public ApiResult<?> deleteBatch(String boModelCode, List<Long> ids) {
        var boModelDo = boModelService.getBoModelVoByCode(boModelCode);
        String tableName = boModelDo.getDatabaseTableName();
        List<DynamicBoFieldDefinitionDO> boFieldDefinitionDoList = getBoFieldDefinitionByBoModelCodeDos(boModelCode);
        if (boFieldDefinitionDoList == null || boFieldDefinitionDoList.isEmpty()) {
            throw new BusinessException("未找到Bo字段");
        }
        var condition = DSL.field(DSL.name(ID)).in(ids);
        //设置租户条件
        setTenantIdCondition(condition);
        Table<?> table = DSL.table(DSL.name(tableName));
        dsl.deleteFrom(table)
                .where(condition)
                .execute();
        return ApiResult.ok();
    }

    @Override
    @Transactional
    public ApiResult<Object> insertBatch(String boModelCode, List<Map<String, Object>> entityDataList) {
        for (Map<String, Object> stringObjectMap : entityDataList) {
            insert(boModelCode,stringObjectMap);
        }
        return ApiResult.ok();
    }



    private static void boFiledCheck( Map<String, DynamicBoFieldDefinitionDO> map,
                                     Map<String, Object> entityData) {

        if (entityData == null || entityData.size() == 0) {
            throw new BusinessException("数据不能为空");
        }
        entityData.forEach((key, value) -> {
            if(!map.containsKey(key)){
                throw new BusinessException("未找到Bo字段："+key);
            }
        });

    }

    /**
     * 返回bo字段中定义属性 需要是数据库字段的
     **/
    private List<DynamicBoFieldDefinitionDO> getBoFieldDefinitionByBoModelCodeDos(String boModelCode) {
//        var boModelDo = boModelService.getBoModelVoByCode(boModelCode);
//        String tableName = boModelDo.getDatabaseTableName();
        List<DynamicBoFieldDefinitionDO> boFieldDefinitionDoList;
        BoFieldQueryParam boFieldParam = new BoFieldQueryParam();
        boFieldParam.setDbField(true);
        boFieldParam.setBasicModuleCode(boModelCode);

//        boFieldParam.setDbTableName(tableName);
        boFieldDefinitionDoList =
                boFieldDefinitionService.searchBoFieldList(boFieldParam);
        return boFieldDefinitionDoList;
    }

    /**
     * 设置租户条件
     **/
    private Condition setTenantIdCondition(Condition condition) {
        var tenantId = dbAuditFieldService.getCurrentTenantId();
        if (tenantId == null) {
            tenantId = -1L;
        }
        //租户条件
        Field<Long> tenantIdField = DSL.field(DSL.name(TENANT_ID), Long.class);
        return condition.and(tenantIdField.eq(tenantId));

    }

    private Map<String, DynamicBoFieldDefinitionDO> buildFieldDefinitionMap(List<DynamicBoFieldDefinitionDO> boFieldDefinitionDoList) {
        return boFieldDefinitionDoList.stream()
                .collect(Collectors.toMap(def -> def.getBasicKey(), def -> def));
    }
}
