package com.elitescloud.cloudt.lowcode.dynamic.service.dynamic.impl;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.db.meta.Column;
import cn.hutool.db.meta.MetaUtil;
import cn.hutool.db.meta.Table;
import cn.hutool.db.meta.TableType;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.util.DatetimeUtil;
import com.elitescloud.boot.util.ExceptionsUtil;
import com.elitescloud.boot.util.ObjUtil;
import com.elitescloud.cloudt.lowcode.dynamic.common.DeployType;
import com.elitescloud.cloudt.lowcode.dynamic.model.bo.DynamicModelColumnBO;
import com.elitescloud.cloudt.lowcode.dynamic.model.bo.DynamicModelTableBO;
import com.elitescloud.cloudt.lowcode.dynamic.model.convert.BoModelFieldTypeEnum;
import com.elitescloud.cloudt.lowcode.dynamic.model.entity.DynamicDeployModelRecordDO;
import com.elitescloud.cloudt.lowcode.dynamic.model.entity.DynamicDeployRecordDO;
import com.elitescloud.cloudt.lowcode.dynamic.repo.DynamicConfigurationRepoProc;
import com.elitescloud.cloudt.lowcode.dynamic.repo.DynamicDeployModelRecordRepoProc;
import com.elitescloud.cloudt.lowcode.dynamic.repo.DynamicDeployRecordRepoProc;
import com.google.common.base.Functions;
import org.jooq.AlterTableStep;
import org.jooq.CreateTableColumnStep;
import org.jooq.DSLContext;
import org.jooq.DataType;
import org.jooq.impl.DSL;
import org.jooq.impl.SQLDataType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.jooq.JooqProperties;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import javax.sql.DataSource;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

/**
 * 数据模型部署服务.
 *
 * @author Kaiser（wang shao）
 * @date 2024/7/25
 */
@Service
public class DynamicBoModelDeployService {
    private static final Logger logger = LoggerFactory.getLogger(DynamicBoModelDeployService.class);

    @Autowired
    private DynamicDeployRecordRepoProc repoProc;
    @Autowired
    private DynamicDeployModelRecordRepoProc modelRecordRepoProc;
    @Autowired
    private DynamicConfigurationRepoProc configurationRepoProc;

    @Autowired
    private TaskExecutor taskExecutor;

    /**
     * 部署数据模型
     *
     * @param deployType       部署类型
     * @param tableBoList      表模型
     * @param configuratorCode 功能模块编码
     */
    @Transactional(rollbackFor = Exception.class)
    public void deploy(@NotNull DeployType deployType, @NotEmpty List<DynamicModelTableBO> tableBoList, String configuratorCode) {
        Assert.notNull(deployType, "未知部署类型");
        Assert.notEmpty(tableBoList, "数据表模型为空");

        // 先保存部署记录
        DynamicDeployRecordDO recordDO = new DynamicDeployRecordDO();
        recordDO.setDynamicConfiguratorCode(ObjUtil.defaultIfNull(configuratorCode, ""));
        recordDO.setDeployType(deployType.name());
        recordDO.setStartTime(LocalDateTime.now());
        recordDO.setSuccess(false);
        recordDO.setFinished(false);
        repoProc.save(recordDO);

        CompletableFuture.runAsync(() -> startDeploy(recordDO, deployType, tableBoList), taskExecutor)
                .whenComplete((r, e) -> {
                    String failReason = null;
                    if (e != null) {
                        failReason = ExceptionsUtil.stackTraceAllToString(e);
                    }
                    try {
                        repoProc.updateDeployResult(recordDO.getId(), failReason);
                        configurationRepoProc.updateDeployed(configuratorCode, deployType != DeployType.CANCEL_DEPLOY);
                    } catch (Exception ex) {
                        logger.error("更新部署结果异常：{}, {}", recordDO.getId(), failReason, ex);
                    }
                });
    }

    private void startDeploy(DynamicDeployRecordDO recordDO, DeployType deployType, List<DynamicModelTableBO> tableBoList) {
        logger.info("开始{}{}...", deployType.getDescription(), recordDO.getDynamicConfiguratorCode());

        var datasource = SpringContextHolder.getObjectProvider(DataSource.class).getIfAvailable();
        org.springframework.util.Assert.notNull(datasource, "获取数据源失败");
        var dslContext = this.buildDsl(datasource);
        var schema = this.getSchemaName(dslContext);

        // 整理待更新的模型
        var tableWrappers = this.analyzeTableWrapper(deployType, tableBoList, datasource, schema);

        for (TableExecutor tableWrapper : tableWrappers) {
            // 执行更新
            tableWrapper.execute(dslContext);

            // 保存记录
            DynamicDeployModelRecordDO deployModelRecordDO = new DynamicDeployModelRecordDO();
            deployModelRecordDO.setRecordId(recordDO.getId());
            deployModelRecordDO.setBoModelCode(tableWrapper.getBoModelCode());
            deployModelRecordDO.setConfiguratorCode(recordDO.getDynamicConfiguratorCode());
            deployModelRecordDO.setLogJson(tableWrapper.getDescription());
            deployModelRecordDO.setPrettyLogJson(tableWrapper.getDescriptionPretty());

            modelRecordRepoProc.save(deployModelRecordDO);
        }
    }

    private List<TableExecutor> analyzeTableWrapper(DeployType deployType, List<DynamicModelTableBO> tableBoList, DataSource dataSource, String schema) {
        var tableNames = MetaUtil.getTables(dataSource, schema, TableType.TABLE);

        List<TableExecutor> wrappers = new ArrayList<>(tableBoList.size());
        for (DynamicModelTableBO tableBO : tableBoList) {
            if (CharSequenceUtil.isBlank(tableBO.getDatabaseTableName())) {
                throw new BusinessException("数据模型" + tableBO.getBoModelCode() + "配置异常，未获取到表名");
            }
            var tableName = tableBO.getDatabaseTableName();

            switch (deployType) {
                case DEPLOY:
                case REDEPLOY:
                    wrappers.add(new TableExecutor(deployType, tableName, tableNames.contains(tableName), tableBO));
                    break;
                case CANCEL_DEPLOY:
                    var tableMetadata = MetaUtil.getTableMeta(dataSource, null, schema, tableName);
                    wrappers.add(new TableExecutor(deployType, tableName, tableNames.contains(tableName), tableBO, tableMetadata));
                    break;
                default:
                    throw new BusinessException("暂不支持的部署类型" + deployType);
            }
        }
        return wrappers;
    }

    private String getSchemaName(DSLContext dslContext) {
        try (var query = dslContext.select(DSL.field("database()", String.class))) {
            var result = query.fetchOne();
            return result == null ? null : result.value1();
        }
    }

    private DSLContext buildDsl(DataSource dataSource) {
        var dialect = new JooqProperties().determineSqlDialect(dataSource);
        return DSL.using(dataSource, dialect);
    }

    public static class TableExecutor {
        private final DeployType deployType;
        private final String tableName;
        private final boolean existsTable;
        private final DynamicModelTableBO tableBo;
        private final Table tableMeta;

        public TableExecutor(DeployType deployType, String tableName, boolean existsTable, DynamicModelTableBO tableBo) {
            this.deployType = deployType;
            this.tableName = tableName;
            this.existsTable = existsTable;
            this.tableBo = tableBo;
            this.tableMeta = null;
        }

        public TableExecutor(DeployType deployType, String tableName, boolean existsTable, DynamicModelTableBO tableBo, Table tableMeta) {
            this.deployType = deployType;
            this.tableName = tableName;
            this.existsTable = existsTable;
            this.tableBo = tableBo;
            this.tableMeta = tableMeta;
        }

        public String getBoModelCode() {
            return tableBo.getBoModelCode();
        }

        public String getDescription() {
            switch (deployType) {
                case DEPLOY:
                    return "部署表" + tableName;
                case REDEPLOY:
                    return "重新部署表" + tableName;
                case CANCEL_DEPLOY:
                    return "取消部署表" + tableName;
            }
            return "未知操作";
        }

        public String getDescriptionPretty() {
            switch (deployType) {
                case DEPLOY:
                    if (existsTable) {
                        return "更新表：" + tableName;
                    }
                    return "新增表：" + tableName;
                case REDEPLOY:
                    return "删除并新增表" + tableName;
                case CANCEL_DEPLOY:
                    return "删除表：" + tableName;
            }
            return "未知操作";
        }

        public void execute(DSLContext dslContext) {
            switch (deployType) {
                case DEPLOY:
                    // 更新部署
                    if (existsTable) {
                        executeUpdateTable(dslContext);
                    } else {
                        executeCreateTable(dslContext);
                    }
                    break;
                case REDEPLOY:
                    // 重新部署，删表后重新建表
                    executeDeleteTable(dslContext);
                    executeCreateTable(dslContext);
                    break;
                case CANCEL_DEPLOY:
                    // 取消部署，删除表
                    executeDeleteTable(dslContext);
                    break;
            }
        }

        private void executeUpdateTable(DSLContext dslContext) {
            AlterTableStep alter = dslContext.alterTable(tableName);

            Map<String, Column> columnMap = tableMeta == null || tableMeta.getColumns() == null ? Collections.emptyMap() :
                    tableMeta.getColumns().stream().collect(Collectors.toMap(Column::getName, Functions.identity(), (t1, t2) -> t1));
            for (DynamicModelColumnBO columnBO : tableBo.getColumnList()) {
                if (columnMap.containsKey(columnBO.getBasicKey())) {
                    // 存在则更新
                    var column = columnMap.get(columnBO.getBasicKey());

                    // 默认值
                    if (!CharSequenceUtil.equals(column.getColumnDef(), columnBO.getBasicDefaultValue())) {
                        if (StringUtils.hasText(columnBO.getBasicDefaultValue())) {
                            try (var step = alter.alterColumn(columnBO.getBasicKey()).setDefault((Object) this.normalizeDefaultValue(columnBO.getBasicType(), columnBO.getBasicDefaultValue()))) {
                                step.execute();
                            }
                        } else {
                            try (var step = alter.alterColumn(columnBO.getBasicKey()).dropDefault()) {
                                step.execute();
                            }
                        }
                    }

                    // 必填
                    if (column.isNullable() && Boolean.FALSE.equals(columnBO.getDbFieldNullable())) {
                        try (var step = alter.alterColumn(columnBO.getBasicKey()).setNotNull()) {
                            step.execute();
                        }
                    } else if (!column.isNullable() && Boolean.TRUE.equals(columnBO.getDbFieldNullable())) {
                        try (var step = alter.alterColumn(columnBO.getBasicKey()).dropNotNull()) {
                            step.execute();
                        }
                    }
                    continue;
                }

                // 不存在则新增
                try (var step = alter.addColumn(columnBO.getBasicKey(), buildDataType(columnBO))) {
                    step.execute();
                }
            }
        }

        private void executeCreateTable(DSLContext dslContext) {
            try (CreateTableColumnStep step = dslContext.createTable(tableName)) {
                List<String> keyColumns = new ArrayList<>(8);
                for (DynamicModelColumnBO columnBO : tableBo.getColumnList()) {
                    step.column(columnBO.getBasicKey(), buildDataType(columnBO));

                    // 主键
                    if (Boolean.TRUE.equals(columnBO.getPrimaryKey())) {
                        keyColumns.add(columnBO.getBasicKey());
                    }
                }
                if (!keyColumns.isEmpty()) {
                    step.constraint(DSL.constraint("pk_" + tableName).primaryKey(keyColumns.toArray(new String[0])));
                }

                step.execute();
            }
        }

        private void executeDeleteTable(DSLContext dslContext) {
            if (!existsTable) {
                // 表不存在
                return;
            }

            try (var query = dslContext.dropTable(tableName)) {
                query.execute();
            }
        }

        private String buildComment(DynamicModelColumnBO columnBO) {
            var description = columnBO.getBasicName();
            if (CharSequenceUtil.isBlank(columnBO.getBasicDescription())) {
                return description;
            }

            return StringUtils.hasText(description) ? description + "," + columnBO.getBasicDescription() : columnBO.getBasicDescription();
        }

        private DataType<?> buildDataType(DynamicModelColumnBO columnBO) {
            DataType<?> dataType = null;
            Object defaultValue = this.normalizeDefaultValue(columnBO.getBasicType(), columnBO.getBasicDefaultValue());

            switch (columnBO.getBasicType()) {
                case STRING:
                    dataType = SQLDataType.VARCHAR(ObjUtil.defaultIfNull(columnBO.getDbFieldLength(), 255));
                    if (defaultValue != null) {
                        ((DataType<String>) dataType).defaultValue((String) defaultValue);
                    }
                    break;
                case BOOLEAN:
                    dataType = SQLDataType.BOOLEAN;
                    if (defaultValue != null) {
                        ((DataType<Boolean>) dataType).defaultValue((Boolean) defaultValue);
                    }
                    break;
                case EMAIL:
                    dataType = SQLDataType.VARCHAR(ObjUtil.defaultIfNull(columnBO.getDbFieldLength(), 255));
                    if (defaultValue != null) {
                        ((DataType<String>) dataType).defaultValue((String) defaultValue);
                    }
                    break;
                case MONEY:
                    dataType = SQLDataType.DECIMAL(ObjUtil.defaultIfNull(columnBO.getDbFieldPrecision(), 18), ObjUtil.defaultIfNull(columnBO.getDbFieldScale(), 6));
                    if (defaultValue != null) {
                        ((DataType<BigDecimal>) dataType).defaultValue((BigDecimal) defaultValue);
                    }
                    break;
//            case BYTE:
//                dataType =  SQLDataType.TINYINT;
//                break;
//            case BINARY:
//                dataType =  SQLDataType.BINARY;
//                break;
                case LONG:
                    dataType = SQLDataType.BIGINT;
                    if (defaultValue != null) {
                        ((DataType<Long>) dataType).defaultValue((Long) defaultValue);
                    }
                    break;
//            case FLOAT:
//                dataType =  SQLDataType.FLOAT;
                case DOUBLE:
                    dataType = SQLDataType.DOUBLE;
                    if (defaultValue != null) {
                        ((DataType<Double>) dataType).defaultValue((Double) defaultValue);
                    }
                    break;
                case INTEGER:
                    dataType = SQLDataType.INTEGER;
                    if (defaultValue != null) {
                        ((DataType<Integer>) dataType).defaultValue((Integer) defaultValue);
                    }
                    break;
//            case DECIMAL:
//                dataType =  SQLDataType.DECIMAL;
//                break;
                case DATE:
                    dataType = SQLDataType.LOCALDATE;
                    if (defaultValue != null) {
                        ((DataType<LocalDate>) dataType).defaultValue((LocalDate) defaultValue);
                    }
                    break;
                case DATE_TIME:
                    dataType = SQLDataType.LOCALDATETIME;
                    if (defaultValue != null) {
                        ((DataType<LocalDateTime>) dataType).defaultValue((LocalDateTime) defaultValue);
                    }
                    break;
                // ... 其他类型的映射 ...
                default:
                    throw new IllegalStateException("Unsupported field type: " + this);
            }

            return dataType
                    .nullable(columnBO.getDbFieldNullable() == null || Boolean.TRUE.equals(columnBO.getDbFieldNullable()));
        }

        private <T> T normalizeDefaultValue(BoModelFieldTypeEnum fieldTypeEnum, String defaultValue) {
            if (!StringUtils.hasText(defaultValue)) {
                return null;
            }

            switch (fieldTypeEnum) {
                case STRING:
                    return (T) defaultValue;
                case BOOLEAN:
                    return (T) Boolean.valueOf(defaultValue);
                case EMAIL:
                    return (T) defaultValue;
                case MONEY:
                    return (T) new BigDecimal(defaultValue);
//            case BYTE:
//                dataType =  SQLDataType.TINYINT;
//                break;
//            case BINARY:
//                dataType =  SQLDataType.BINARY;
//                break;
                case LONG:
                    return (T) Long.valueOf(defaultValue);
//            case FLOAT:
//                dataType =  SQLDataType.FLOAT;
                case DOUBLE:
                    return (T) Double.valueOf(defaultValue);
                case INTEGER:
                    return (T) Integer.valueOf(defaultValue);
//            case DECIMAL:
//                dataType =  SQLDataType.DECIMAL;
//                break;
                case DATE:
                    return (T) DatetimeUtil.parseLocalDate(defaultValue);
                case DATE_TIME:
                    return (T) DatetimeUtil.parseLocalDateTime(defaultValue);
                // ... 其他类型的映射 ...
                default:
                    throw new IllegalStateException("Unsupported field type: " + this);
            }
        }
    }
}
