package com.elitescloud.cloudt.lowcode.dynamic.service.spi.impl;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.lowcode.dynamic.common.WorkflowEngineStatusUdc;
import com.elitescloud.cloudt.lowcode.dynamic.service.spi.DynamicWorkflowEngine;
import com.elitescloud.cloudt.lowcode.dynamic.service.spi.common.ProcDefInfo;
import com.elitescloud.cloudt.lowcode.dynamic.service.spi.common.WorkflowStatus;
import com.elitescloud.cloudt.workflow.params.ProcDefDTO;
import com.elitescloud.cloudt.workflow.params.save.ProcDefSaveDTO;
import com.elitescloud.cloudt.workflow.service.rpc.ProcDefRpcService;
import com.elitescloud.cloudt.workflow.service.rpc.WorkflowRpcService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/7/24
 */
@Component
public class CloudtDynamicWorkflowEngine implements DynamicWorkflowEngine {
    private static final Logger logger = LoggerFactory.getLogger(CloudtDynamicWorkflowEngine.class);

    @Autowired
    private WorkflowRpcService workflowRpcService;
    @Autowired
    private ProcDefRpcService procDefRpcService;
    @Autowired
    private UdcProvider udcProvider;

    @Override
    public String engineType() {
        return "cloudt";
    }

    @Override
    public String engineName() {
        return "云梯工作流";
    }

    @Override
    public WorkflowStatus getEngineStatus() {
        WorkflowStatus workflowStatus = new WorkflowStatus();

        try {
            workflowRpcService.ready();

            workflowStatus.setStatus(WorkflowEngineStatusUdc.RUNNING.getValue());
            workflowStatus.setStatusName(this.translateWorkflowStatusName(WorkflowEngineStatusUdc.RUNNING));
        } catch (Exception e) {
            logger.error("连接工作流异常：", e);
            workflowStatus.setStatus(WorkflowEngineStatusUdc.CONNECTION_EXP.getValue());
            workflowStatus.setStatusName(this.translateWorkflowStatusName(WorkflowEngineStatusUdc.CONNECTION_EXP));
            workflowStatus.setExpMsg(e.getMessage());
        }
        return workflowStatus;
    }

    @Override
    public ProcDefInfo getProcDefInfo(String procDefKey) {
        ProcDefDTO procDef = null;
        try {
            procDef = procDefRpcService.getProcDef(procDefKey).computeData();
        } catch (Exception e) {
            throw new BusinessException("查询流程定义失败", e);
        }
        if (procDef == null) {
            return null;
        }

        ProcDefInfo info = new ProcDefInfo();
        info.setId(procDef.getId().toString());
        info.setKey(procDef.getKey());
        info.setCategoryId(procDef.getCategoryId() == null ? null : procDef.getCategoryId().toString());
        info.setCategoryName(procDef.getCategoryName());
        info.setName(procDef.getName());
        info.setStatus(procDef.getState());
        info.setStatusName(procDef.getStateName());
        info.setAppPrefixUrl(procDef.getAppPrefixUrl());
        return info;
    }

    @Override
    public String createProcDef(ProcDefInfo procDefInfo) {
        ProcDefSaveDTO saveDTO = new ProcDefSaveDTO();
        saveDTO.setId(StringUtils.hasText(procDefInfo.getId()) ? Long.parseLong(procDefInfo.getId()) : null);
        saveDTO.setCategoryId(StringUtils.hasText(procDefInfo.getCategoryId()) ? Long.parseLong(procDefInfo.getCategoryId()) : null);
        saveDTO.setName(procDefInfo.getName());
        saveDTO.setKey(procDefInfo.getKey());
        saveDTO.setAppPrefixUrl(procDefInfo.getAppPrefixUrl());
        saveDTO.setForm(procDefInfo.getForm());
        saveDTO.setModuleCode(procDefInfo.getModuleCode());
        saveDTO.setModuleName(procDefInfo.getModuleName());

        try {
            procDefRpcService.save(saveDTO);
        } catch (Exception e) {
            throw new BusinessException("保存流程定义失败", e);
        }

        return procDefInfo.getKey();
    }

    @Override
    public String deploy(String procDefKey) {
        try {
            procDefRpcService.deploy(procDefKey);
        } catch (Exception e) {
            throw new BusinessException("部署流程失败：" + e.getMessage(), e);
        }
        return procDefKey;
    }

    @Override
    public String deleteProcDef(String procDefKey) {
        try {
            procDefRpcService.deleteByKey(procDefKey);
        } catch (Exception e) {
            throw new BusinessException("删除流程失败:" + e.getMessage(), e);
        }
        return procDefKey;
    }

    private String translateWorkflowStatusName(WorkflowEngineStatusUdc status) {
        var udcValueMap = udcProvider.getValueMapByUdcCode(status.getAppCode(), status.getUdcCode());
        if (CollUtil.isEmpty(udcValueMap)) {
            return status.getDescription();
        }

        return udcValueMap.getOrDefault(status.getValue(), status.getDescription());
    }
}
