package com.elitescloud.cloudt.lowcode.dynamic.repo;

import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.lowcode.dynamic.model.bo.DynamicWorkflowCfgSimpleBO;
import com.elitescloud.cloudt.lowcode.dynamic.model.entity.DynamicConfiguratorDO;
import com.elitescloud.cloudt.lowcode.dynamic.model.entity.QDynamicConfiguratorDO;
import com.elitescloud.cloudt.lowcode.dynamic.model.vo.query.DynamicConfiguratorPageQueryVO;
import com.elitescloud.cloudt.lowcode.dynamic.model.vo.resp.DynamicConfiguratorPageRespVO;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.dsl.StringExpression;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import javax.validation.constraints.NotBlank;
import java.util.Map;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/7/22
 */
@Repository
public class DynamicConfigurationRepoProc extends BaseRepoProc<DynamicConfiguratorDO> {
    private static final QDynamicConfiguratorDO QDO = QDynamicConfiguratorDO.dynamicConfiguratorDO;

    public DynamicConfigurationRepoProc() {
        super(QDO);
    }

    /**
     * 更新部署流程
     *
     * @param id
     * @param deployed
     */
    @Transactional(rollbackFor = Exception.class)
    public void updateDeployedApproval(long id, boolean deployed) {
        super.jpaQueryFactory.update(QDO)
                .set(QDO.deployedApproval, deployed)
                .where(QDO.id.eq(id))
                .execute();
    }

    /**
     * 删除流程
     *
     * @param id
     */
    public void deleteApproval(long id) {
        super.jpaQueryFactory.update(QDO)
                .set(QDO.createdApproval, false)
                .set(QDO.deployedApproval, false)
                .setNull(QDO.approvalEngine)
                .setNull(QDO.approvalKey)
                .setNull(QDO.approvalJson)
                .where(QDO.id.eq(id))
                .execute();
    }

    /**
     * 更新部署状态
     *
     * @param id
     * @param deployed
     */
    @Transactional(rollbackFor = Exception.class)
    public void updateDeployed(long id, boolean deployed) {
        super.jpaQueryFactory.update(QDO)
                .set(QDO.deployed, deployed)
                .where(QDO.id.eq(id))
                .execute();
    }

    /**
     * 更新部署状态
     *
     * @param code
     * @param deployed
     */
    @Transactional(rollbackFor = Exception.class)
    public void updateDeployed(String code, boolean deployed) {
        super.jpaQueryFactory.update(QDO)
                .set(QDO.deployed, deployed)
                .where(QDO.dynamicConfiguratorCode.eq(code))
                .execute();
    }

    /**
     * 更新为已发布
     *
     * @param id
     */
    public void updatePublished(long id, String appCode, String menuCode) {
        super.jpaQueryFactory.update(QDO)
                .set(QDO.published, true)
                .set(QDO.appCode, appCode)
                .set(QDO.menuCode, menuCode)
                .where(QDO.id.eq(id))
                .execute();
    }

    /**
     * 取消发布
     *
     * @param id
     */
    public void revokePublish(long id) {
        super.jpaQueryFactory.update(QDO)
                .set(QDO.published, false)
                .setNull(QDO.appCode)
                .setNull(QDO.menuCode)
                .where(QDO.id.eq(id))
                .execute();
    }

    /**
     * 更新表单配置
     *
     * @param id
     * @param formCfg
     */
    public void updateFormCfg(long id, Map<String, Object> formCfg) {
        super.jpaQueryFactory.update(QDO)
                .set(QDO.formJson, formCfg)
                .where(QDO.id.eq(id))
                .execute();
    }

    /**
     * 更新列表配置
     *
     * @param id
     * @param listPageCfg
     */
    public void updateListPageCfg(long id, Map<String, Object> listPageCfg) {
        super.jpaQueryFactory.update(QDO)
                .set(QDO.listPageJson, listPageCfg)
                .where(QDO.id.eq(id))
                .execute();
    }

    /**
     * 更新审批配置
     *
     * @param id
     * @param created
     * @param approvalType
     * @param approvalKey
     * @param approvalJson
     */
    public void updateApprovalCfg(long id, boolean created, String approvalType, String approvalEngine, String approvalKey,
                                  Map<String, Object> approvalJson) {
        super.jpaQueryFactory.update(QDO)
                .set(QDO.createdApproval, created)
                .set(QDO.approvalType, approvalType)
                .set(QDO.approvalEngine, approvalEngine)
                .set(QDO.approvalKey, approvalKey)
                .set(QDO.approvalJson, approvalJson)
                .where(QDO.id.eq(id))
                .execute();
    }

    /**
     * 编码是否存在
     *
     * @param code
     * @param id
     * @return
     */
    public boolean existsCode(@NotBlank String code, Long id) {
        return super.exists(QDO.dynamicConfiguratorCode, code, id);
    }

    /**
     * 分页查询管理
     *
     * @param queryVO
     * @return
     */
    public PagingVO<DynamicConfiguratorPageRespVO> pageMng(DynamicConfiguratorPageQueryVO queryVO) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.dynamicConfiguratorCode, queryVO.getDynamicConfiguratorCode())
                .andLike(QDO.dynamicConfiguratorName, queryVO.getDynamicConfiguratorName())
                .andEq(QDO.approvalType, queryVO.getApprovalType())
                .andEq(QDO.enabled, queryVO.getEnabled())
                .andEq(QDO.published, queryVO.getPublished())
                .andEq(QDO.deployed, queryVO.getDeployed())
                .andLike(new StringExpression[]{QDO.dynamicConfiguratorCode, QDO.dynamicConfiguratorName}, queryVO.getKeyword())
                .build();
        var qbean = Projections.bean(DynamicConfiguratorPageRespVO.class, QDO.id, QDO.dynamicConfiguratorCode, QDO.dynamicConfiguratorName, QDO.dynamicConfiguratorDescription,
                QDO.enabled, QDO.approvalType, QDO.published, QDO.appCode, QDO.menuCode, QDO.deployed,
                QDO.createTime, QDO.creator, QDO.modifyTime, QDO.updater);
        return super.queryByPage(qbean, predicate, queryVO.getPageRequest());
    }

    /**
     * 是否部署
     *
     * @param id
     * @return
     */
    public Boolean isDeployed(long id) {
        return super.getValue(QDO.deployed, id);
    }

    /**
     * 是否已发布
     *
     * @param id
     * @return
     */
    public Boolean isPublished(long id) {
        return super.getValue(QDO.published, id);
    }

    /**
     * 获取发布的菜单编码
     *
     * @param id
     * @return
     */
    public String getMenuCode(long id) {
        return super.getValue(QDO.menuCode, id);
    }

    /**
     * 获取编码
     *
     * @param id
     * @return
     */
    public String getCode(long id) {
        return super.getValue(QDO.dynamicConfiguratorCode, id);
    }

    /**
     * 获取编码
     *
     * @param id
     * @return
     */
    public String getName(long id) {
        return super.getValue(QDO.dynamicConfiguratorName, id);
    }

    /**
     * 获取部署状态
     *
     * @param id
     * @return
     */
    public Boolean getDeploy(long id) {
        return super.getValue(QDO.deployed, id);
    }

    /**
     * 获取发布状态
     *
     * @param id
     * @return
     */
    public Boolean getPublished(long id) {
        return super.getValue(QDO.published, id);
    }

    /**
     * 获取表单配置
     *
     * @param id
     * @return
     */
    public Map<String, Object> getFormJson(long id) {
        return super.getValue(QDO.formJson, id);
    }

    /**
     * 获取列表配置
     *
     * @param id
     * @return
     */
    public Map<String, Object> getListPageJson(long id) {
        return super.getValue(QDO.listPageJson, id);
    }

    /**
     * 获取流程配置
     *
     * @param id
     * @return
     */
    public DynamicWorkflowCfgSimpleBO getWorkflowCfgSimpleBO(long id) {
        var qbean = Projections.bean(DynamicWorkflowCfgSimpleBO.class, QDO.createdApproval, QDO.deployedApproval, QDO.approvalType, QDO.approvalEngine,
                QDO.approvalKey, QDO.approvalJson);
        return super.get(qbean, id);
    }
}
