package com.elitescloud.cloudt.lowcode.dynamic.service.dynamic.impl;

import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.lowcode.dynamic.model.entity.DynamicBoFieldDefinitionDO;
import com.elitescloud.cloudt.lowcode.dynamic.model.param.BoFieldQueryParam;
import com.elitescloud.cloudt.lowcode.dynamic.model.param.DynamicExportParam;
import com.elitescloud.cloudt.lowcode.dynamic.model.param.DynamicFieldQueryParam;
import com.elitescloud.cloudt.lowcode.dynamic.model.vo.DynamicImportFailures;
import com.elitescloud.cloudt.lowcode.dynamic.model.vo.DynamicImportResult;
import com.elitescloud.cloudt.lowcode.dynamic.service.db.DbAuditFieldService;
import com.elitescloud.cloudt.lowcode.dynamic.service.dynamic.DynamicBoModelService;
import com.elitescloud.cloudt.lowcode.dynamic.service.dynamic.DynamicDbApiService;
import com.elitescloud.cloudt.lowcode.dynamic.service.dynamic.DynamicUtil;
import lombok.extern.slf4j.Slf4j;
import org.jooq.*;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author : chen.niu
 * @description :
 * @date : 2024-04-02 19:36
 */
@Service
@Slf4j
public class DynamicDbApiServiceImpl implements DynamicDbApiService {
    public static final String TENANT_ID = "tenant_id";
    public static final String ID = "id";
    //    @Autowired
//    private ApplicationContext applicationContext;

    @Autowired
    DSLContext dsl;
//    @Autowired
//    DbTableService dbTableService;

    @Autowired
    DynamicBoFieldDefinitionServiceImpl boFieldDefinitionService;
    @Autowired
    DbAuditFieldService dbAuditFieldService;

    @Autowired
    DynamicBoModelService boModelService;

    @Override
    public ApiResult<Object> insert(String tableName,
                                    Map<String, Object> entityData) throws Exception {
        List<DynamicBoFieldDefinitionDO> boFieldDefinitionDoList = getBoFieldDefinitionDos(tableName);
        boFiledCheck(boFieldDefinitionDoList, entityData);

        // 首先，创建一个动态表引用
        Table<?> table = DSL.table(DSL.name(tableName));
        // 创建字段和值的列表
        List<Field<?>> fields = new ArrayList<>();
        List<Object> values = new ArrayList<>();

        dbAuditFieldService.fillInsertAuditFields(entityData);
        // 填充字段和值的列表
        entityData.forEach((key, value) -> {
            fields.add(DSL.field(DSL.name(key)));
            values.add(value);
        });

        // 使用fields()和values()方法来构建插入步骤
        InsertValuesStepN<?> insertStep = dsl
                .insertInto(table)
                .columns(fields)
                .values(values);
        String sql = insertStep.getSQL();
        System.out.println(sql);
        // 执行插入
        insertStep.execute();
        return ApiResult.ok(entityData.get("id"));
    }


    @Override
    public ApiResult<Integer> update(String tableName, Long id,
                                     Map<String, Object> entityData) {
        List<DynamicBoFieldDefinitionDO> boFieldDefinitionDoList = getBoFieldDefinitionDos(tableName);
        boFiledCheck(boFieldDefinitionDoList, entityData);
        Table<?> table = DSL.table(DSL.name(tableName));
        // 初始化更新查询
        UpdateQuery<?> updateQuery = dsl.updateQuery(table);
        dbAuditFieldService.fillUpdateAuditFields(entityData);
        // 动态添加更新字段和值
        entityData.forEach((key, value) -> {
            Field<Object> field = DSL.field(DSL.name(key), Object.class);
            updateQuery.addValue(field, value);
        });

        // 应用WHERE条件，这里假设表中有一个名为"id"的字段用作主键
        Field<Long> idField = DSL.field(DSL.name(ID), Long.class);

        Condition condition = idField.eq(id);
        //租户条件
        condition = setTenantIdCondition(condition);
        updateQuery.addConditions(condition);
        // 执行更新操作
        int rowsUpdated = updateQuery.execute();

        return ApiResult.ok(rowsUpdated);
    }


    @Override
    public ApiResult<Page<?>> queryPage(String tableName,
                                        DynamicFieldQueryParam queryParam) {
        Table<?> table = DSL.table(DSL.name(tableName));
        // 构建基本查询
        SelectJoinStep<?> baseQuery = dsl.select().from(table);
//        SelectConditionStep<?> query = dsl
//                .select() // 这里可以具体指定需要选择的字段
//                .from(table)
//                .join(DSL.table(DSL.name("other_table"))) // 示例：添加一个JOIN到其他表
//                .on(DSL.field(DSL.name("table.field")).eq(DSL.field(DSL.name("other_table.other_field")))) // 指定JOIN条件
//                .where(DSL.field(DSL.name("table.condition_field")).eq("someValue")); // 添加WHERE条件

        // 构造条件查询
        Condition condition = DynamicUtil.getDynamicFieldQueryParamCondition(queryParam);
        //租户条件
        condition = setTenantIdCondition(condition);
        SelectConditionStep<?> conditionalQuery = baseQuery.where(condition);
        // 应用排序
        var sortFields = queryParam.getOrders().stream()
                .map(order -> {
                    Field<Object> orderField = DSL.field(DSL.name(order.getColumn()), Object.class);
                    return order.isAsc() ? orderField.asc() : orderField.desc();
                })
                .toArray(SortField[]::new);
        conditionalQuery.orderBy(sortFields);

        // 应用分页并获取结果
        int offset = queryParam.getCurrent() * queryParam.getSize();
        Result<?> result = conditionalQuery
                .limit(queryParam.getSize())
                .offset(offset)
                .fetch();

        // 计算总数，重用相同的条件
        int total = dsl.selectCount().from(table).where(condition).fetchOne(0, int.class);

        // 将JOOQ结果转换为Page对象
        Page<?> pageResult = new PageImpl<>(result.intoMaps(),
                PageRequest.of(queryParam.getCurrent(), queryParam.getSize()), total);


        return ApiResult.ok(pageResult);
    }


    @Override
    public ApiResult<Map> getById(String tableName,
                                  Long id) {
        try {
            org.jooq.Table<?> table = DSL.table(DSL.name(tableName));
//            // 获取表对应的Table对象
//            if (table == null) {
//                // 如果表不存在，返回错误信息
//                return ApiResult.fail("Table not found: " + tableName);
//            }
            // 创建查询条件
            org.jooq.Condition condition = DSL.field(DSL.name(ID)).eq(id);
            //设置租户条件
            condition = setTenantIdCondition(condition);
            // 执行查询
            org.jooq.Record record = dsl.selectFrom(table).where(condition).fetchOne();
            if (record == null) {
                return ApiResult.fail("未找到数据");
            }
            Map<String, Object> map = record.intoMap();
            return ApiResult.ok(map);
        } catch (Exception e) {
            // 处理异常
            log.error("动态查询异常：", e);
            return ApiResult.fail("动态查询异常 data by ID: " + e.getMessage());
        }
    }

    @Override
    public void dynamicExport(String tableName, DynamicExportParam dynamicExportParam, HttpServletResponse response) throws IOException {
        if (dynamicExportParam == null || dynamicExportParam.getFieldMap() == null) {
            throw new RuntimeException("参数不能为空");
        }

        // 动态构建双层表头
        List<List<String>> head = new ArrayList<>();
        dynamicExportParam.getFieldMap().forEach((englishName, chineseName) -> {
            List<String> headerColumn = new ArrayList<>();
            headerColumn.add(englishName); // 第一行英文
            headerColumn.add(chineseName); // 第二行中文
            head.add(headerColumn);
        });

        // 构建查询字段
        List<Field<?>> selectFields = dynamicExportParam.getFieldMap().keySet().stream()
                .map(DSL::field)
                .collect(Collectors.toList());
        Table<?> table = DSL.table(DSL.name(tableName));

        var condition = DynamicUtil.getDynamicFieldQueryParamCondition(dynamicExportParam);
        //设置租户条件
        condition = setTenantIdCondition(condition);
        //查询执行
        Result<?> result = dsl.select(selectFields)
                .from(table)
                .where(condition)
                .orderBy(dynamicExportParam.getOrders().stream()
                        .map(order -> DSL.field(DSL.name(order.getColumn()), Object.class)
                                .sort(order.isAsc() ? SortOrder.ASC : SortOrder.DESC))
                        .toArray(SortField[]::new)).fetch();
        List<List<Object>> dataList = new ArrayList<>();
        if (result != null && result.size() > 0) {
            // 转换数据为 List<List<String>>，其中内部列表代表一行数据
            result.intoMaps().stream().map(map -> map.entrySet().stream()
                            .sorted(Map.Entry.comparingByKey()) // 可能需要根据实际需要对字段排序，以确保顺序与表头对应
                            .map(stringObjectEntry -> {
                                if (Timestamp.class.isAssignableFrom(stringObjectEntry.getClass())) {
                                    return ((Timestamp) stringObjectEntry.getValue()).toLocalDateTime();
                                }
                                return stringObjectEntry.getValue();
                            })
                            .collect(Collectors.toList()))
                    .collect(Collectors.toList());
        }

        try {
            DynamicUtil.writeResponseXlsx(tableName, response, head, dataList);
        } catch (Exception e) {
            log.error("导出Excel失败", e);
            throw e;
        }
    }


    @Override
    public ApiResult<DynamicImportResult> dynamicImport(String tableName,
                                                        MultipartFile file) {
        DynamicImportResult dynamicImportResult = new DynamicImportResult();
        List<Map<String, Object>> dataList = null;
        try {
            //读取excel数据
            dataList = DynamicUtil.readExcelAsMap(file);

        } catch (Exception e) {
            return ApiResult.fail("文件读取失败" + e.getMessage());
        }


        // 获取数据库表对象
        // Table<?> table = dsl.table(tableName);
        Table table = DSL.table(DSL.name(tableName));
        int row=0;
        // 动态构建插入语句
        for (Map<String, Object> rowData : dataList) {
            row++;
            try {
                InsertSetStep<org.jooq.Record> insertStep = dsl.insertInto(table);
                dbAuditFieldService.fillInsertAuditFields(rowData);
                rowData.forEach((key, value) -> {
                    insertStep.set(DSL.field(key), value);
                });
                insertStep.columns().execute();
            } catch (Exception e) {
                DynamicImportFailures dynamicImportFailures= new DynamicImportFailures();
                dynamicImportFailures.setFailuresDate(rowData);
                dynamicImportFailures.setFailuresDescription("執行异常："+e.getMessage());
                dynamicImportFailures.setFailuresRow(row);

                dynamicImportResult.getFailuresDescription().add(dynamicImportFailures);
                dynamicImportResult.setFailures(dynamicImportResult.getFailures() + 1);
                continue;
            }
            dynamicImportResult.setSuccessful(dynamicImportResult.getSuccessful() + 1);
        }

        return ApiResult.ok(dynamicImportResult);

    }

    @Override
    public ApiResult<?> delete(String tableName, Long id) {
        List<DynamicBoFieldDefinitionDO> boFieldDefinitionDoList = getBoFieldDefinitionDos(tableName);
        if (boFieldDefinitionDoList == null || boFieldDefinitionDoList.isEmpty()) {
            throw new BusinessException("未找到Bo字段");
        }
        var condition = DSL.field(DSL.name(ID)).eq(id);
        //设置租户条件
        setTenantIdCondition(condition);
        Table<?> table = DSL.table(DSL.name(tableName));
        dsl.deleteFrom(table)
                .where(condition)
                .execute();
        return ApiResult.ok();
    }
    @Override
    public ApiResult<?> deleteBatch(String tableName, List<Long> ids) {
        List<DynamicBoFieldDefinitionDO> boFieldDefinitionDoList = getBoFieldDefinitionDos(tableName);
        if (boFieldDefinitionDoList == null || boFieldDefinitionDoList.isEmpty()) {
            throw new BusinessException("未找到Bo字段");
        }
        var condition = DSL.field(DSL.name(ID)).in(ids);
        //设置租户条件
        setTenantIdCondition(condition);
        Table<?> table = DSL.table(DSL.name(tableName));
        dsl.deleteFrom(table)
                .where(condition)
                .execute();
        return ApiResult.ok();
    }

    private static void boFiledCheck(List<DynamicBoFieldDefinitionDO> boFieldDefinitionDoList, Map<String, Object> entityData) {
        if (boFieldDefinitionDoList == null || boFieldDefinitionDoList.isEmpty()) {
            throw new BusinessException("未找到Bo字段");
        }
        if (entityData == null || entityData.size() == 0) {
            throw new BusinessException("数据不能为空");
        }
    }

    /**
     * 返回bo字段中定义属性 需要是数据库字段的
     **/
    private List<DynamicBoFieldDefinitionDO> getBoFieldDefinitionDos(String tableName) {
        List<DynamicBoFieldDefinitionDO> boFieldDefinitionDoList;
        BoFieldQueryParam boFieldParam = new BoFieldQueryParam();
        boFieldParam.setDbField(true);
        boFieldParam.setDbTableName(tableName);
        boFieldDefinitionDoList =
                boFieldDefinitionService.searchBoFieldList(boFieldParam);
        return boFieldDefinitionDoList;
    }

    /**
     * 设置租户条件
     **/
    private Condition setTenantIdCondition(Condition condition) {
        var tenantId = dbAuditFieldService.getCurrentTenantId();
        if (tenantId == null) {
            tenantId = -1L;
        }
        //租户条件
        Field<Long> tenantIdField = DSL.field(DSL.name(TENANT_ID), Long.class);
        return condition.and(tenantIdField.eq(tenantId));

    }
}
