package com.elitescloud.cloudt.lowcode.dynamic.controller;

import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.lowcode.dynamic.model.convert.BoModelFieldTypeEnum;
import com.elitescloud.cloudt.lowcode.dynamic.model.vo.query.DynamicConfiguratorPageQueryVO;
import com.elitescloud.cloudt.lowcode.dynamic.model.vo.resp.*;
import com.elitescloud.cloudt.lowcode.dynamic.model.vo.save.*;
import com.elitescloud.cloudt.lowcode.dynamic.service.dynamic.DynamicConfigurationMngService;
import com.elitescloud.cloudt.lowcode.dynamic.service.spi.common.WorkflowStatus;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 功能模块管理.
 *
 * @author Kaiser（wang shao）
 * @date 2024/7/22
 */
@Api(tags = "功能模块管理")
@RestController
@RequestMapping(value = "/mng/dynamicConfigurator", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
public class DynamicConfiguratorMngController {
    private DynamicConfigurationMngService mngService;

    @ApiOperation(value = "保存功能模块信息")
    @ApiOperationSupport(order = 1)
    @PostMapping(value = "/save")
    public ApiResult<Long> saveDynamicConfigurator(@Valid @RequestBody DynamicConfiguratorSaveVO saveVO) {
        return mngService.saveDynamicConfigurator(saveVO);
    }

    @ApiOperation(value = "分页查询管理")
    @ApiOperationSupport(order = 2)
    @PostMapping(value = "/page")
    public ApiResult<PagingVO<DynamicConfiguratorPageRespVO>> pageQuery(@RequestBody DynamicConfiguratorPageQueryVO queryVO) {
        return mngService.pageQuery(queryVO);
    }

    @ApiOperation(value = "获取功能模块详细信息")
    @ApiOperationSupport(order = 3)
    @ApiImplicitParam(name = "id", value = "功能模块的ID", dataType = "long", required = true)
    @GetMapping(value = "/detail")
    public ApiResult<DynamicConfiguratorDetailRespVO> getDynamicConfigurator(@RequestParam(name = "id") @NotNull(message = "ID为空") Long id) {
        return mngService.getDynamicConfigurator(id);
    }

    @ApiOperation(value = "发布功能模块")
    @ApiOperationSupport(order = 4)
    @PostMapping(value = "/publish")
    public ApiResult<Long> publishDynamicConfigurator(@RequestBody @Valid DynamicConfiguratorPublishSaveVO saveVO) {
        return mngService.publishDynamicConfigurator(saveVO);
    }

    @ApiOperation(value = "取消发布功能模块")
    @ApiOperationSupport(order = 5)
    @ApiImplicitParam(name = "id", value = "功能模块的ID", dataType = "long", required = true)
    @PostMapping(value = "/revokePublish")
    public ApiResult<Long> revokePublishDynamicConfigurator(@RequestParam(name = "id") @NotNull(message = "ID为空") Long id) {
        return mngService.revokePublishDynamicConfigurator(id);
    }

    @ApiOperation(value = "删除功能模块")
    @ApiOperationSupport(order = 6)
    @ApiImplicitParam(name = "id", value = "功能模块的ID", dataType = "long", required = true)
    @DeleteMapping(value = "/delete")
    public ApiResult<Long> deletePublishDynamicConfigurator(@RequestParam(name = "id") @NotNull(message = "ID为空") Long id) {
        return mngService.deleteDynamicConfigurator(id);
    }

    @ApiOperation(value = "获取功能模块下的数据模型列表")
    @ApiOperationSupport(order = 11)
    @ApiImplicitParam(name = "id", value = "功能模块的ID", dataType = "long", required = true)
    @GetMapping(value = "/boModel/list")
    public ApiResult<List<DynamicConfiguratorBoModelListRespVO>> listBoModel(@RequestParam(name = "id") @NotNull(message = "ID为空") Long id) {
        return mngService.listBoModel(id);
    }

    @ApiOperation(value = "获取功能模块下的数据模型详细列表")
    @ApiOperationSupport(order = 11)
    @ApiImplicitParam(name = "id", value = "功能模块的ID", dataType = "long", required = true)
    @GetMapping(value = "/boModel/listDetail")
    public ApiResult<List<DynamicConfiguratorBoModelDetailListRespVO>> listBoModelDetail(@RequestParam(name = "id") @NotNull(message = "ID为空") Long id) {
        return mngService.listBoModelDetail(id);
    }

    @ApiOperation(value = "保存数据模型")
    @ApiOperationSupport(order = 12)
    @PostMapping(value = "/boModel/save")
    public ApiResult<Long> saveBoModel(@RequestBody @Valid DynamicBoModelSaveVO saveVO) {
        return mngService.saveBoModel(saveVO);
    }

    @ApiOperation(value = "获取数据模型信息")
    @ApiOperationSupport(order = 13)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "功能模块的ID", dataType = "long", required = true),
            @ApiImplicitParam(name = "boModelId", value = "数据模型的ID", dataType = "long", required = true),
    })
    @GetMapping(value = "/boModel/detail")
    public ApiResult<DynamicConfiguratorBoModelDetailRespVO> getBoModel(@RequestParam(name = "id") @NotNull(message = "ID为空") Long id,
                                                                        @RequestParam(name = "boModelId") @NotNull(message = "数据模型ID为空") Long boModelId) {
        return mngService.getBoModel(id, boModelId);
    }

    @ApiOperation(value = "删除数据模型")
    @ApiOperationSupport(order = 14)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "功能模块的ID", dataType = "long", required = true),
            @ApiImplicitParam(name = "boModelId", value = "数据模型的ID", dataType = "long", required = true),
            @ApiImplicitParam(name = "deleteModel", value = "是否同时删除模型", dataType = "boolean", defaultValue = "true"),
    })
    @DeleteMapping(value = "/boModel/delete")
    public ApiResult<Long> deleteBoModel(@RequestParam(name = "id") @NotNull(message = "功能模块ID为空") Long id,
                                         @RequestParam(name = "boModelId") @NotNull(message = "数据模型ID为空") Long boModelId,
                                         @RequestParam(name = "deleteModel", required = false, defaultValue = "true") Boolean deleteModel) {
        return mngService.deleteBoModel(id, boModelId, deleteModel);
    }

    @ApiOperation(value = "获取模型的字段列表")
    @ApiOperationSupport(order = 21)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "boModelId", value = "数据模型的ID", dataType = "long", required = true),
    })
    @GetMapping(value = "/boModel/field/list")
    public ApiResult<List<DynamicConfiguratorBoFieldListRespVO>> getBoFieldList(@RequestParam(name = "boModelId") @NotNull(message = "数据模型ID为空") Long boModelId) {
        return mngService.getBoFieldList(boModelId);
    }

    @ApiOperation(value = "新增模型字段")
    @ApiOperationSupport(order = 22)
    @ApiImplicitParam(name = "boModelId", value = "数据模型的ID", dataType = "long", required = true)
    @PostMapping(value = "/boModel/field/add")
    public ApiResult<Long> addBoModelField(@RequestParam(name = "boModelId") @NotNull(message = "数据模型ID为空") Long boModelId,
                                           @RequestBody @Valid DynamicBoFieldSaveVO fieldSaveVO) {
        return mngService.addBoModelField(boModelId, fieldSaveVO);
    }

    @ApiOperation(value = "删除模型字段")
    @ApiOperationSupport(order = 23)
    @ApiImplicitParam(name = "boFieldId", value = "字段ID", dataType = "long", required = true)
    @DeleteMapping(value = "/boModel/field/delete")
    public ApiResult<Long> deleteBoModelField(@RequestParam(name = "boFieldId") @NotNull(message = "字段ID为空") Long boFieldId) {
        return mngService.deleteBoModelField(boFieldId);
    }

    @ApiOperation(value = "保存字段列表")
    @ApiOperationSupport(order = 24)
    @ApiImplicitParam(name = "boModelId", value = "数据模型的ID", dataType = "long", required = true)
    @PostMapping(value = "/boModel/field/save")
    public ApiResult<Long> saveBoModelFields(@RequestParam(name = "boModelId") @NotNull(message = "数据模型ID为空") Long boModelId,
                                             @RequestBody @Valid List<DynamicBoFieldSaveVO> fieldSaveList) {
        return mngService.saveBoModelFields(boModelId, fieldSaveList);
    }

    @ApiOperation(value = "获取字段详情")
    @ApiOperationSupport(order = 25)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "boFieldId", value = "字段ID", dataType = "long", required = true),
    })
    @GetMapping(value = "/boModel/field/detail")
    public ApiResult<DynamicConfiguratorBoFieldDetailRespVO> getBoFieldDetail(@RequestParam(name = "boFieldId") @NotNull(message = "字段ID为空") Long boFieldId) {
        return mngService.getBoFieldDetail(boFieldId);
    }

    @ApiOperation(value = "获取字段类型列表")
    @ApiOperationSupport(order = 26)
    @GetMapping(value = "/boModel/field/enum/dataType")
    public ApiResult<List<CodeNameParam>> getFieldTypeList() {
        var valueList = Arrays.stream(BoModelFieldTypeEnum.values()).map(t -> new CodeNameParam(t.name(), t.getValueDescription())).collect(Collectors.toList());
        return ApiResult.ok(valueList);
    }

    @ApiOperation(value = "获取表单配置")
    @ApiOperationSupport(order = 31)
    @ApiImplicitParam(name = "id", value = "功能模块的ID", dataType = "long", required = true)
    @GetMapping(value = "/formCfg")
    public ApiResult<Map<String, Object>> getFormCfg(@RequestParam(name = "id") @NotNull(message = "功能模块ID为空") Long id) {
        return mngService.getFormCfg(id);
    }

    @ApiOperation(value = "保存表单配置")
    @ApiOperationSupport(order = 32)
    @PostMapping(value = "/formCfg")
    public ApiResult<Long> saveFormCfg(@RequestBody @Valid DynamicConfiguratorFormSaveVO formSaveVO) {
        return mngService.saveFormCfg(formSaveVO);
    }

    @ApiOperation(value = "获取列表配置")
    @ApiOperationSupport(order = 33)
    @ApiImplicitParam(name = "id", value = "功能模块的ID", dataType = "long", required = true)
    @GetMapping(value = "/listPageCfg")
    public ApiResult<Map<String, Object>> getListPageCfg(@RequestParam(name = "id") @NotNull(message = "功能模块ID为空") Long id) {
        return mngService.getListPageCfg(id);
    }

    @ApiOperation(value = "保存列表配置")
    @ApiOperationSupport(order = 34)
    @PostMapping(value = "/listPageCfg")
    public ApiResult<Long> saveListPageCfg(@RequestBody @Valid DynamicConfiguratorListPageSaveVO listPageSaveVO) {
        return mngService.saveListPageCfg(listPageSaveVO);
    }

    @ApiOperation(value = "获取审批设计信息")
    @ApiOperationSupport(order = 41)
    @ApiImplicitParam(name = "id", value = "功能模块的ID", dataType = "long", required = true)
    @GetMapping(value = "/approval/info")
    public ApiResult<DynamicConfiguratorApprovalRespVO> getApprovalInfo(@RequestParam(name = "id") @NotNull(message = "功能模块ID为空") Long id) {
        return mngService.getApprovalInfo(id);
    }

    @ApiOperation(value = "获取工作流引擎列表")
    @ApiOperationSupport(order = 42)
    @GetMapping(value = "/approval/engine/list")
    public ApiResult<List<CodeNameParam>> getApprovalEngineList() {
        return mngService.getApprovalEngineList();
    }

    @ApiOperation(value = "获取工作流引擎状态")
    @ApiOperationSupport(order = 43)
    @ApiImplicitParam(name = "engineType", value = "工作流引擎类型", required = true)
    @GetMapping(value = "/approval/engine/status")
    public ApiResult<WorkflowStatus> getApprovalEngineStatus(@NotBlank(message = "工作流引擎类型为空") String engineType) {
        return mngService.getApprovalEngineStatus(engineType);
    }

    @ApiOperation(value = "保存审批设计信息")
    @ApiOperationSupport(order = 44)
    @PostMapping(value = "/approval/save")
    public ApiResult<Long> saveApproval(@RequestBody @Valid DynamicConfiguratorApprovalSaveVO saveVO) {
        return mngService.saveApproval(saveVO);
    }

    @ApiOperation(value = "部署审批流程")
    @ApiOperationSupport(order = 45)
    @ApiImplicitParam(name = "id", value = "功能模块的ID", dataType = "long", required = true)
    @PostMapping(value = "/approval/deploy")
    public ApiResult<Long> deployApproval(@RequestParam(name = "id") @NotNull(message = "功能模块ID为空") Long id) {
        return mngService.deployApproval(id);
    }

    @ApiOperation(value = "删除审批流程")
    @ApiOperationSupport(order = 46)
    @ApiImplicitParam(name = "id", value = "功能模块的ID", dataType = "long", required = true)
    @DeleteMapping(value = "/approval")
    public ApiResult<Long> deleteApproval(@RequestParam(name = "id") @NotNull(message = "功能模块ID为空") Long id) {
        return mngService.deleteApproval(id);
    }

    @ApiOperation(value = "部署功能模块")
    @ApiOperationSupport(order = 51)
    @ApiImplicitParam(name = "id", value = "功能模块的ID", dataType = "long", required = true)
    @PostMapping(value = "/deploy")
    public ApiResult<Long> deploy(@RequestParam(name = "id") @NotNull(message = "功能模块ID为空") Long id) {
        return mngService.deploy(id);
    }

    @ApiOperation(value = "重新部署功能模块")
    @ApiOperationSupport(order = 52)
    @ApiImplicitParam(name = "id", value = "功能模块的ID", dataType = "long", required = true)
    @PostMapping(value = "/redeploy")
    public ApiResult<Long> redeploy(@RequestParam(name = "id") @NotNull(message = "功能模块ID为空") Long id) {
        return mngService.redeploy(id);
    }

    @ApiOperation(value = "取消部署功能模块")
    @ApiOperationSupport(order = 52)
    @ApiImplicitParam(name = "id", value = "功能模块的ID", dataType = "long", required = true)
    @PostMapping(value = "/cancelDeploy")
    public ApiResult<Long> cancelDeploy(@RequestParam(name = "id") @NotNull(message = "功能模块ID为空") Long id) {
        return mngService.cancelDeploy(id);
    }

    @ApiOperation(value = "获取部署信息")
    @ApiOperationSupport(order = 53)
    @ApiImplicitParam(name = "id", value = "功能模块的ID", dataType = "long", required = true)
    @GetMapping(value = "/deploy/info")
    public ApiResult<DynamicDeployInfoRespVO> getDeployInfo(@RequestParam(name = "id") @NotNull(message = "功能模块ID为空") Long id) {
        return mngService.getDeployInfo(id);
    }

    @Autowired
    public void setMngService(DynamicConfigurationMngService mngService) {
        this.mngService = mngService;
    }
}
