package com.elitescloud.cloudt.lowcode.dynamic.service.db.impl;

import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.lowcode.dynamic.model.entity.DbFieldDo;
import com.elitescloud.cloudt.lowcode.dynamic.model.entity.DbTableDo;
import com.elitescloud.cloudt.lowcode.dynamic.repo.DbFieldRepository;
import com.elitescloud.cloudt.lowcode.dynamic.repo.DbTableRepository;
import com.elitescloud.cloudt.lowcode.dynamic.service.db.DbMetadataService;
import lombok.extern.slf4j.Slf4j;
import org.jooq.*;
import org.jooq.impl.DSL;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author : chen.niu
 * @description :
 * @date : 2024-04-16 10:54
 */
@Service
@Slf4j
public class DbJooqMysqlMetadataServiceImpl implements DbMetadataService {

    private final org.jooq.DSLContext dsl;


    public DbJooqMysqlMetadataServiceImpl(DSLContext dsl ) {

        this.dsl = dsl;

    }

    @Override
    public List<DbFieldDo> getDbTableFieldDo(String schema, String tableName) {
        if (tableName == null || tableName.isEmpty()) {
            return new ArrayList<>();
        }
        var records = dsl.select(
                        DSL.field(DbFieldConstants.FIELD_TABLE_CATALOG),
                        DSL.field(DbFieldConstants.FIELD_TABLE_SCHEMA),
                        DSL.field(DbFieldConstants.FIELD_TABLE_NAME),
                        DSL.field(DbFieldConstants.FIELD_COLUMN_NAME),
                        DSL.field(DbFieldConstants.FIELD_DATA_TYPE),
                        DSL.field(DbFieldConstants.FIELD_CHARACTER_MAXIMUM_LENGTH),
                        DSL.field(DbFieldConstants.ORDINAL_POSITION),
                        DSL.field(DbFieldConstants.FIELD_NUMERIC_PRECISION),
                        DSL.field(DbFieldConstants.FIELD_NUMERIC_SCALE),
                        DSL.field(DbFieldConstants.FIELD_IS_NULLABLE),
                        DSL.field(DbFieldConstants.FIELD_COLUMN_DEFAULT),
                        DSL.field(DbFieldConstants.FIELD_COLUMN_KEY),
                        DSL.field(DbFieldConstants.FIELD_EXTRA),
                        DSL.field(DbFieldConstants.FIELD_COLUMN_COMMENT)
                )
                .from(DSL.table(DbFieldConstants.FROM_TABLE_NAME))
                .where(DSL.field(DbFieldConstants.FIELD_TABLE_NAME).eq(tableName))
                .fetch();
        if (records == null || records.size() == 0) {
            return null;
        }
        List<DbFieldDo> dbFields = records.stream().map(this::mapRecordToDbFieldDo).collect(Collectors.toList());
        return dbFields;
    }

    @Override
    public List<DbTableDo> getDbTableDo(String schema, String tableName) {
        if (schema == null) {
            schema = getCurrentSchema();
            log.info("schema is null,use current schema:{}", schema);
        }
        var select = dsl.select(
                DSL.field(DatabaseConstants.TABLE_CATALOG),
                DSL.field(DatabaseConstants.TABLE_SCHEMA),
                DSL.field(DatabaseConstants.TABLE_NAME),
                DSL.field(DatabaseConstants.TABLE_TYPE),
                DSL.field(DatabaseConstants.ENGINE),
                DSL.field(DatabaseConstants.VERSION),
                DSL.field(DatabaseConstants.ROW_FORMAT),
                DSL.field(DatabaseConstants.TABLE_ROWS),
                DSL.field(DatabaseConstants.AVG_ROW_LENGTH),
                DSL.field(DatabaseConstants.DATA_LENGTH),
                DSL.field(DatabaseConstants.MAX_DATA_LENGTH),
                DSL.field(DatabaseConstants.INDEX_LENGTH),
                DSL.field(DatabaseConstants.DATA_FREE),
                DSL.field(DatabaseConstants.AUTO_INCREMENT),
                DSL.field(DatabaseConstants.CREATE_TIME),
                DSL.field(DatabaseConstants.UPDATE_TIME),
                DSL.field(DatabaseConstants.CHECK_TIME),
                DSL.field(DatabaseConstants.TABLE_COLLATION),
                DSL.field(DatabaseConstants.CHECKSUM),
                DSL.field(DatabaseConstants.CREATE_OPTIONS),
                DSL.field(DatabaseConstants.TABLE_COMMENT)
        ).from(DatabaseConstants.FROM_TABLE_NAME);
        Result<Record21<Object, Object, Object, Object, Object, Object,
                Object, Object, Object, Object, Object, Object, Object, Object, Object,
                Object, Object, Object, Object, Object, Object>> tables = null;

        if (tableName != null) {
            tables = select.where(DSL.field(DbFieldConstants.FIELD_TABLE_NAME).eq(tableName))
                    .and(DSL.field(DatabaseConstants.TABLE_SCHEMA).eq(schema))
                    .fetch();
        } else {
            tables = select.where(DSL.field(DatabaseConstants.TABLE_SCHEMA).eq(schema)).fetch();
        }
        List<DbTableDo> dbTableDos = new ArrayList<>();
        for (org.jooq.Record record : tables) {
            DbTableDo dbTable = new DbTableDo();
            dbTable.setTableCatalog(record.get(DatabaseConstants.TABLE_CATALOG, String.class));
            dbTable.setTableSchema(record.get(DatabaseConstants.TABLE_SCHEMA, String.class));
            dbTable.setTableName(record.get(DatabaseConstants.TABLE_NAME, String.class));
            dbTable.setTableType(record.get(DatabaseConstants.TABLE_TYPE, String.class));
            dbTable.setEngine(record.get(DatabaseConstants.ENGINE, String.class));
            dbTable.setVersion(record.get(DatabaseConstants.VERSION, Integer.class));
            dbTable.setRowFormat(record.get(DatabaseConstants.ROW_FORMAT, String.class));
            dbTable.setTableRows(record.get(DatabaseConstants.TABLE_ROWS, Long.class));
            dbTable.setAvgRowLength(record.get(DatabaseConstants.AVG_ROW_LENGTH, Long.class));
            dbTable.setDataLength(record.get(DatabaseConstants.DATA_LENGTH, Long.class));
            dbTable.setMaxDataLength(record.get(DatabaseConstants.MAX_DATA_LENGTH, Long.class));
            dbTable.setIndexLength(record.get(DatabaseConstants.INDEX_LENGTH, Long.class));
            dbTable.setDataFree(record.get(DatabaseConstants.DATA_FREE, Long.class));
            dbTable.setAutoIncrement(record.get(DatabaseConstants.AUTO_INCREMENT, Long.class));
            Timestamp createTimeTimestamp = record.get(DatabaseConstants.CREATE_TIME, Timestamp.class);
            dbTable.setCreateTimeTable(createTimeTimestamp != null ? createTimeTimestamp.toLocalDateTime() : null);
            Timestamp updateTimeTimestamp = record.get(DatabaseConstants.UPDATE_TIME, Timestamp.class);
            dbTable.setUpdateTimeTable(updateTimeTimestamp != null ? updateTimeTimestamp.toLocalDateTime() : null);
            Timestamp checkTimeTimestamp = record.get(DatabaseConstants.CHECK_TIME, Timestamp.class);
            dbTable.setCheckTimeTable(checkTimeTimestamp != null ? checkTimeTimestamp.toLocalDateTime() : null);

            dbTable.setTableCollation(record.get(DatabaseConstants.TABLE_COLLATION, String.class));
            dbTable.setChecksum(record.get(DatabaseConstants.CHECKSUM, Long.class));
            dbTable.setCreateOptions(record.get(DatabaseConstants.CREATE_OPTIONS, String.class));
            dbTable.setTableComment(record.get(DatabaseConstants.TABLE_COMMENT, String.class));
            dbTableDos.add(dbTable);
        }
        return dbTableDos;
    }


    @Override
    public String getCurrentSchema() {
        Result<Record1<String>> result = dsl.select(DSL.field("DATABASE()", String.class)).fetch();
        if (result.isNotEmpty()) {
            return result.get(0).value1();
        } else {
            // 处理查询结果为空的情况
            return null;
        }
    }


    @Override
    public List<String> getAllSchemas() {
        return dsl.select().from("information_schema.SCHEMATA").fetch().stream().map(record -> record.get("SCHEMA_NAME", String.class))
                .collect(Collectors.toList());
    }


    private DbFieldDo mapRecordToDbFieldDo(org.jooq.Record record) {
        DbFieldDo field = new DbFieldDo();
        field.setTableCatalog(record.get(DbFieldConstants.FIELD_TABLE_CATALOG, String.class));
        field.setDbTableName(record.get(DbFieldConstants.FIELD_TABLE_NAME, String.class));
        field.setTableSchema(record.get(DbFieldConstants.FIELD_TABLE_SCHEMA, String.class));
        field.setDbFieldName(record.get(DbFieldConstants.FIELD_COLUMN_NAME, String.class));
        field.setDbFieldType(record.get(DbFieldConstants.FIELD_DATA_TYPE, String.class));
        field.setDbOrder(record.get(DbFieldConstants.ORDINAL_POSITION, Integer.class));
        field.setDbFieldLength(record.get(DbFieldConstants.FIELD_CHARACTER_MAXIMUM_LENGTH, Integer.class));
        field.setDbFieldPrecision(record.get(DbFieldConstants.FIELD_NUMERIC_PRECISION, Integer.class));
        field.setDbFieldScale(record.get(DbFieldConstants.FIELD_NUMERIC_SCALE, Integer.class));
        field.setDbFieldNullable("YES".equals(record.get(DbFieldConstants.FIELD_IS_NULLABLE, String.class)));
        field.setDbFieldDefaultValue(record.get(DbFieldConstants.FIELD_COLUMN_DEFAULT, String.class));
        field.setDbFieldPrimaryKey("PRI".equals(record.get(DbFieldConstants.FIELD_COLUMN_KEY, String.class)));
        field.setDbFieldUnique("UNI".equals(record.get(DbFieldConstants.FIELD_COLUMN_KEY, String.class)));
        field.setDbFieldIndex("MUL".equals(record.get(DbFieldConstants.FIELD_COLUMN_KEY, String.class)));
        field.setDbFieldAutoIncrement("auto_increment".equals(record.get(DbFieldConstants.FIELD_EXTRA, String.class)));
        field.setDbFieldDescription(record.get(DbFieldConstants.FIELD_COLUMN_COMMENT, String.class));
        return field;
    }

    public static class DatabaseConstants {
        public static final String FROM_TABLE_NAME = "INFORMATION_SCHEMA.tables"; //查询元数据的表名
        public static final String TABLE_CATALOG = "TABLE_CATALOG";  // 数据库的目录名
        public static final String TABLE_SCHEMA = "TABLE_SCHEMA";  // 数据库的模式名
        public static final String TABLE_NAME = "TABLE_NAME";  // 表的名称
        public static final String TABLE_TYPE = "TABLE_TYPE";  // 表的类型
        public static final String ENGINE = "ENGINE";  // 使用的存储引擎
        public static final String VERSION = "VERSION";  // 表的版本
        public static final String ROW_FORMAT = "ROW_FORMAT";  // 行格式
        public static final String TABLE_ROWS = "TABLE_ROWS";  // 表中的行数
        public static final String AVG_ROW_LENGTH = "AVG_ROW_LENGTH";  // 平均行长度
        public static final String DATA_LENGTH = "DATA_LENGTH";  // 数据长度
        public static final String MAX_DATA_LENGTH = "MAX_DATA_LENGTH";  // 最大数据长度
        public static final String INDEX_LENGTH = "INDEX_LENGTH";  // 索引长度
        public static final String DATA_FREE = "DATA_FREE";  // 空闲数据空间
        public static final String AUTO_INCREMENT = "AUTO_INCREMENT";  // 下一个自增值
        public static final String CREATE_TIME = "CREATE_TIME";  // 表的创建时间
        public static final String UPDATE_TIME = "UPDATE_TIME";  // 表的最后更新时间
        public static final String CHECK_TIME = "CHECK_TIME";  // 表的最后检查时间
        public static final String TABLE_COLLATION = "TABLE_COLLATION";  // 表的字符集和校对规则
        public static final String CHECKSUM = "CHECKSUM";  // 表的校验和
        public static final String CREATE_OPTIONS = "CREATE_OPTIONS";  // 创建表时的其他选项
        public static final String TABLE_COMMENT = "TABLE_COMMENT";  // 表的注释

    }

    public static class DbFieldConstants {
        public static final String FROM_TABLE_NAME = "INFORMATION_SCHEMA.COLUMNS";
        // 用于 SQL 查询中的字段名称常量，指向 information_schema 数据库。
        public static final String FIELD_TABLE_NAME = "TABLE_NAME"; // 表示表的名称。
        public static final String FIELD_COLUMN_NAME = "COLUMN_NAME"; // 表示列的名称。
        public static final String FIELD_DATA_TYPE = "COLUMN_TYPE"; // 表示列的数据类型。
        public static final String ORDINAL_POSITION = "ORDINAL_POSITION"; //排序。

        public static final String FIELD_CHARACTER_MAXIMUM_LENGTH = "CHARACTER_MAXIMUM_LENGTH"; // 表示字符类型列的最大长度。
        public static final String FIELD_NUMERIC_PRECISION = "NUMERIC_PRECISION"; // 表示数值列的数值精度。
        public static final String FIELD_NUMERIC_SCALE = "NUMERIC_SCALE"; // 表示小数类型列的小数位数。
        public static final String FIELD_IS_NULLABLE = "IS_NULLABLE"; // 表示该列是否可以存储 null 值。
        public static final String FIELD_COLUMN_DEFAULT = "COLUMN_DEFAULT"; // 表示列的默认值。
        public static final String FIELD_COLUMN_KEY = "COLUMN_KEY"; // 表示列的键类型（如主键、唯一键等）。
        public static final String FIELD_EXTRA = "EXTRA"; // 表示列的额外信息（如是否自增等）。
        public static final String FIELD_COLUMN_COMMENT = "COLUMN_COMMENT"; // 表示列的注释信息。
        public static final String FIELD_TABLE_SCHEMA = "TABLE_SCHEMA";
        public static final String FIELD_TABLE_CATALOG = "TABLE_CATALOG";
    }
}
