package com.elitescloud.cloudt.platform.controller.api;

import com.elitescloud.boot.common.annotation.BusinessObject;
import com.elitescloud.boot.common.annotation.BusinessObjectOperation;
import com.elitescloud.boot.common.annotation.businessobject.OperationTypeEnum;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.platform.model.entity.SysPlatformNumberRuleDO;
import com.elitescloud.cloudt.platform.model.params.seq.SysPlatformNumberRuleAddParam;
import com.elitescloud.cloudt.platform.model.params.seq.SysPlatformNumberRuleDtlVO;
import com.elitescloud.cloudt.platform.model.params.seq.SysPlatformNumberRuleQParam;
import com.elitescloud.cloudt.platform.model.params.seq.SysPlatformNumberRuleVO;
import com.elitescloud.cloudt.platform.service.ISysPlatformNumberRuleService;
import com.elitescloud.cloudt.system.constant.BusinessObjectConstant;
import com.elitescloud.cloudt.system.constant.SysNumType;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import lombok.val;
import org.redisson.api.RScript;
import org.redisson.api.RedissonClient;
import org.redisson.client.codec.StringCodec;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 发号器规则管理
 *
 * @author Michael Li
 * 2020-10-11
 */
@RestController
@RequestMapping("/sys/platform/numbering")
@Api(value = "发号器-规则管理", tags = {"发号器-规则管理"})
@BusinessObject(businessType = BusinessObjectConstant.SYS_PLATFORM_NUMBER_RULE, businessDoClass = SysPlatformNumberRuleDO.class)
public class SysPlatformNumberRuleController {

    private final ISysPlatformNumberRuleService iSysPlatformNumberRuleService;

    private final RedissonClient redissonClient;

    public SysPlatformNumberRuleController(ISysPlatformNumberRuleService sysNumberRuleService, RedissonClient redissonClient) {
        this.iSysPlatformNumberRuleService = sysNumberRuleService;
        this.redissonClient = redissonClient;
    }

    /**
     * Redisson通过对象桶bucket写入redis的例子, bucket使用默认的codec编解码器
     * bucket的name对应redis中的key
     *
     * @return 默认的ok结果
     */
    @GetMapping("/redisson/set")
    @ApiOperation("Redisson-write")
    public ApiResult<Object> setRedisson() {
        val bucket = redissonClient.getBucket("redisson:demo:ruledtl");
        bucket.set(new SysPlatformNumberRuleDtlVO());

        return ApiResult.ok();
    }

    /**
     * Redisson通过对象桶bucket读取redis中的编码规则对象，bucket使用默认的codec解码器
     * 存入和取出的codec需要一样的编解码器，否则会出现乱码或失败异常，例如用redisTemplate设置的value，
     * 用redisson取出因为codec不一致会乱码
     *
     * @return ApiResult携带成功反序列化的编码规则对象
     */
    @GetMapping("/redisson/get")
    @ApiOperation("Redisson-readobj")
    public ApiResult<Object> readRedisson() {
        val bucket = redissonClient.getBucket("redisson:demo:numrule");
        val result = bucket.get();

        return ApiResult.ok(result);
    }

    /**
     * Redisson执行lua脚本的例子，包括入参以及取得返回值
     * 1. 在getScript获取script对象时，需要设置codec用于传参和取值时的正确序列化，才能解析出参数传给lua脚本
     * 2. script.eval函数，需要注意最后两个入参，分别传入keys和values
     * 其中keys是通过字符串数组传入，在lua脚本中用KEYS[n]来取值
     * values是通过varargs的方式传入，在lua脚本中用ARGV[n]来取值
     *
     * @param name 通过uri路径传入的动态参数
     * @return 返回在redis里执行lua脚本后的结果
     */
    @GetMapping("/redisson/script/{name}")
    @ApiOperation("Redisson-script")
    public ApiResult<Object> redissonScript(@PathVariable String name) {
        val script = redissonClient.getScript(StringCodec.INSTANCE);
        val s = "return 'Hello, ' .. ARGV[1]";

        val result = script.eval(
                RScript.Mode.READ_WRITE,
                s,
                RScript.ReturnType.VALUE,
                Collections.emptyList(),
                name
        );

        return ApiResult.ok(result);
    }

    @GetMapping("/redisson/script/calc/{x}")
    @ApiOperation("redisson-script-calc")
    public ApiResult<Object> redissonScriptCalc(@PathVariable String x) {
        val script = redissonClient.getScript(StringCodec.INSTANCE);
        val s = "local a=tonumber(ARGV[1])\nlocal b=tonumber(ARGV[2])\nreturn a+b";

        val result = script.eval(
                RScript.Mode.READ_WRITE,
                s,
                RScript.ReturnType.VALUE,
                Collections.emptyList(),
                x, "99"
        );
        // 验证电话号码是否正确


        return ApiResult.ok(result);
    }

    @GetMapping("/redisson/numseg:")
    @ApiOperation("redisson-号段管理")
    public ApiResult<Object> redissonNumSeg() {
        val script = redissonClient.getScript(StringCodec.INSTANCE);
        val name = "redisson:num:seg";

        val s = "local max = redis.pcall(\"HGET\", KEYS[1], \"max\")\n" +
                "local cur = redis.pcall(\"HGET\", KEYS[1], \"cur\")\n" +
                "if tonumber(cur) >= tonumber(max) then\n" +
                "  \tlocal step = ARGV[1]\n" +
                "  \tif (step == nil) then\n" +
                "    \t\t\tstep = redis.pcall(\"HGET\", KEYS[1], \"step\")\n" +
                "  \tend\n" +
                "  \tredis.pcall(\"HSET\", KEYS[1], \"max\", tonumber(max) + tonumber(step))\n" +
                "end\n" +
                "return redis.pcall(\"HINCRBY\", KEYS[1], \"cur\", 1)";

        val result = script.eval(
                RScript.Mode.READ_WRITE,
                s,
                RScript.ReturnType.VALUE,
                List.of(name),
                5
        );

        return ApiResult.ok(result);
    }

    /**
     * 创建发号器规则
     *
     * @param rule 发号器规则
     * @return 创建结果
     */
    @PostMapping("/rules")
    @ApiOperation("创建发号器规则，成功则返回对应ID")
    @BusinessObjectOperation(
            operationType = OperationTypeEnum.ADD, operationDescription = "创建发号器规则",
            logEnabled = true, argsJsonEnabled = true, resultJsonEnabled = true
    )
    public ApiResult<Long> create(@RequestBody @Valid SysPlatformNumberRuleAddParam rule) {
        return iSysPlatformNumberRuleService.create(rule);
    }

    /**
     * 更新发号器规则
     *
     * @param rule 发号器规则
     * @return 更新结果
     */
    @PutMapping("/rules")
    @ApiOperation("更新发号器规则")
    @BusinessObjectOperation(
            operationType = OperationTypeEnum.UPDATE, operationDescription = "更新发号器规则"
            , logEnabled = true, argsJsonEnabled = true, resultJsonEnabled = true
    )
    public ApiResult<Long> update(@RequestBody @Valid SysPlatformNumberRuleAddParam rule) {
        return iSysPlatformNumberRuleService.update(rule);
    }

    /**
     * 检索发号器规则
     *
     * @param param 查询参数
     * @return 发号器规则
     */
    @PostMapping("/q")
    @ApiOperation("检索发号器规则")
    @BusinessObjectOperation(
            operationType = OperationTypeEnum.SELECT_PAGE, operationDescription = "检索发号器规则"
    )
    public ApiResult<PagingVO<SysPlatformNumberRuleVO>> search(@RequestBody SysPlatformNumberRuleQParam param) {
        return iSysPlatformNumberRuleService.search(param);
    }

    /**
     * 删除发号器规则
     *
     * @param id 发号器规则ID
     * @return 删除结果
     */
    @DeleteMapping("/rules/{id}")
    @ApiOperation("删除发号器规则")
    @BusinessObjectOperation(
            operationType = OperationTypeEnum.DELETE, operationDescription = "删除发号器规则",
            logEnabled = true, argsJsonEnabled = true, resultJsonEnabled = true
    )
    public ApiResult<Boolean> remove(@PathVariable("id") Long id) {
        return iSysPlatformNumberRuleService.removeById(id);
    }

    /**
     * 获取规则详细信息
     *
     * @param id 规则ID
     * @return 规则详细
     */
    @GetMapping("/rules/{id}")
    @ApiOperation("根据规则ID，获取规则对象，包括规则明细列表")
    @BusinessObjectOperation(
            operationType = OperationTypeEnum.SELECT, operationDescription = "获取规则详细"
    )
    public ApiResult<SysPlatformNumberRuleVO> one(@PathVariable("id") Long id) {
        return iSysPlatformNumberRuleService.oneCombined(id);
    }

    /**
     * 更新启用状态
     * <p>
     * 自动取反，如果当前是启用，则修改为禁用，反之修改为启用；
     *
     * @param id 发号规则ID
     * @return 发号规则ID
     */
    @ApiOperation(value = "更新启用状态", notes = "自动取反，如果当前是启用，则修改为禁用，反之修改为启用")
    @ApiImplicitParam(name = "id", value = "发号规则ID", required = true)
    @PatchMapping("/update/{id}/enabled")
    @BusinessObjectOperation(
            operationType = OperationTypeEnum.UPDATE, operationDescription = "更新启用状态",
            logEnabled = true, argsJsonEnabled = true, resultJsonEnabled = true
    )
    public ApiResult<Long> updateEnabled(@PathVariable Long id) {
        return iSysPlatformNumberRuleService.updateEnabled(id);
    }

    /**
     * 取号类型
     *
     * @return 取号类型列表
     */
    @ApiOperation(value = "取号类型")
    @ApiOperationSupport(order = 7)
    @GetMapping(value = "/numType")
    @BusinessObjectOperation(
            operationType = OperationTypeEnum.SELECT, operationDescription = "取号类型"
    )
    public ApiResult<List<CodeNameParam>> numType() {
        var values = Arrays.stream(SysNumType.values())
                .map(t -> new CodeNameParam(t.name(), t.getDescription()))
                .collect(Collectors.toList());
        return ApiResult.ok(values);
    }
}
