package com.elitescloud.cloudt.platform.service.impl;

import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.platform.convert.SysPlatformCurrencyConvert;
import com.elitescloud.cloudt.platform.model.entity.SysPlatformCurrencyDO;
import com.elitescloud.cloudt.platform.model.vo.extend.query.CurrencyPageQueryVO;
import com.elitescloud.cloudt.platform.model.vo.extend.resp.CurrencyDetailRespVO;
import com.elitescloud.cloudt.platform.model.vo.extend.resp.CurrencyPageRespVO;
import com.elitescloud.cloudt.platform.model.vo.extend.save.CurrencySaveVO;
import com.elitescloud.cloudt.platform.service.SysPlatformCurrencyService;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformCurrencyRepo;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformCurrencyRepoProc;
import com.elitescloud.cloudt.system.cacheable.SysCacheCurrencyRpcService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/1/29
 */
@Service
public class SysPlatformCurrencyServiceImpl implements SysPlatformCurrencyService {
    private static final SysPlatformCurrencyConvert CONVERT = SysPlatformCurrencyConvert.INSTANCE;

    @Autowired
    private SysPlatformCurrencyRepo currencyRepo;
    @Autowired
    private SysPlatformCurrencyRepoProc currencyRepoProc;

    @Autowired
    private SysCacheCurrencyRpcService cacheCurrencyRpcService;

    @Override
    public ApiResult<PagingVO<CurrencyPageRespVO>> page(CurrencyPageQueryVO queryVO) {
        var result = currencyRepoProc.pageMng(queryVO)
                .map(CONVERT::do2PageRespVO);
        return ApiResult.ok(result);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> save(CurrencySaveVO saveVO) {
        SysPlatformCurrencyDO currencyDO = null;
        try {
            currencyDO = saveVO.getId() == null ? this.convertForInsert(saveVO) : this.convertForUpdate(saveVO);
        } catch (IllegalArgumentException e) {
            return ApiResult.fail("保存失败，" + e.getMessage());
        }

        currencyRepo.save(currencyDO);

        cacheCurrencyRpcService.clearCacheForAllTenant();
        return ApiResult.ok(currencyDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> delete(Long id) {
        currencyRepoProc.delete(id);

        cacheCurrencyRpcService.clearCacheForAllTenant();
        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> switchEnabled(long id) {
        var enabled = currencyRepoProc.getEnabled(id);
        enabled = enabled == null || !enabled;

        currencyRepoProc.updateEnabled(id, enabled);

        cacheCurrencyRpcService.clearCacheForAllTenant();
        return ApiResult.ok(id);
    }

    @Override
    public ApiResult<CurrencyDetailRespVO> get(Long id) {
        return currencyRepoProc.getOptional(id)
                .map(CONVERT::do2DetailRespVO)
                .map(ApiResult::ok)
                .orElse(ApiResult.noData());
    }

    private SysPlatformCurrencyDO convertForInsert(CurrencySaveVO saveVO) {
        var currencyDO = CONVERT.saveVo2Do(saveVO);

        // 货币编码
        Assert.hasText(saveVO.getCurrCode(), "货币编码为空");
        var exists = currencyRepoProc.existsCode(saveVO.getCurrCode(), null);
        Assert.isTrue(!exists, "货币编码已存在");

        if (currencyDO.getDispDecimal() == null) {
            currencyDO.setDispDecimal(0);
        }
        Assert.isTrue(currencyDO.getDispDecimal() >= 0, "小数位数不能小于0");

        if (currencyDO.getEnabled() == null) {
            currencyDO.setEnabled(false);
        }

        return currencyDO;
    }

    private SysPlatformCurrencyDO convertForUpdate(CurrencySaveVO saveVO) {
        var currencyDO = currencyRepoProc.get(saveVO.getId());

        // 货币编码
        Assert.hasText(saveVO.getCurrCode(), "货币编码为空");
        Assert.isTrue(currencyDO.getCurrCode().equals(saveVO.getCurrCode()), "货币编码不可修改");

        CONVERT.copySaveVo2Do(saveVO, currencyDO);

        if (currencyDO.getDispDecimal() == null) {
            currencyDO.setDispDecimal(0);
        }
        Assert.isTrue(currencyDO.getDispDecimal() >= 0, "小数位数不能小于0");

        if (currencyDO.getEnabled() == null) {
            currencyDO.setEnabled(false);
        }

        return currencyDO;
    }
}
