package com.elitescloud.cloudt.platform.controller.api;

import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.platform.service.SysPlatformDatabaseSourceService;
import com.elitescloud.cloudt.system.datasource.*;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotNull;
import java.util.List;

/**
 * @author : chen.niu
 * @description :
 * @date : 2024-04-29 14:21
 */
@Api(tags = "平台数据库源管理")
@RestController
@RequestMapping("/api/platform/datasource")
@Validated
public class SysPlatformDatabaseSourceController {

    @Autowired
    private SysPlatformDatabaseSourceService sysPlatformDatabaseSourceService;

    @ApiOperation("查询平台数据库源信息")
    @PostMapping("/query")
    public ApiResult<List<SysPlatformDatabaseSourceVO>> queryDatabaseSources(
            @ApiParam(value = "查询参数对象", required = true)
            @Valid @RequestBody SysPlatformDatabaseSourceQueryParam queryParam) {
        return sysPlatformDatabaseSourceService.query(queryParam);
    }

    @ApiOperation("根据ID获取数据库源信息")
    @GetMapping("/getById/{id}")
    public ApiResult<SysPlatformDatabaseSourceVO> getDatabaseSourceById(
            @ApiParam(value = "数据库源的ID", required = true)
            @PathVariable("id")   Long id) {
        return sysPlatformDatabaseSourceService.getById(id);
    }
    @ApiOperation("根据ID获取数据库源密码")
    @GetMapping("/queryPasswordById/{id}")
    public ApiResult<String> queryPasswordById(
            @ApiParam(value = "根据ID获取数据库源密码", required = true)
            @PathVariable("id")   Long id) {
        return sysPlatformDatabaseSourceService.queryPasswordById(id);
    }

    @ApiOperation("根据数据库源代码获取信息")
    @GetMapping("/getByDbsCode")
    public ApiResult<SysPlatformDatabaseSourceVO> getDatabaseSourceByDbsCode(
            @ApiParam(value = "数据库源的代码", required = true)
            @RequestParam("dbsCode") @NotNull String dbsCode) {
        return sysPlatformDatabaseSourceService.getByDbsCode(dbsCode);
    }

    @ApiOperation("保存新的数据库源信息")
    @PostMapping("/save")
    public ApiResult<Long> saveDatabaseSource(
            @ApiParam(value = "包含要保存的数据库源信息的参数对象", required = true)
            @Valid @RequestBody SysPlatformDatabaseSourceSaveParam saveParam) {
        return sysPlatformDatabaseSourceService.save(saveParam);
    }

    @ApiOperation("更新现有的数据库源信息")
    @PutMapping("/update/{id}")
    public ApiResult<Long> updateDatabaseSource(
            @ApiParam(value = "要更新的数据库源ID", required = true)
            @PathVariable("id") @NotNull Long id,
            @ApiParam(value = "包含更新内容的参数对象", required = true)
            @Valid @RequestBody SysPlatformDatabaseSourceUpParam upParam) {
        return sysPlatformDatabaseSourceService.update(id, upParam);
    }

    @ApiOperation("根据ID删除数据库源信息")
    @DeleteMapping("/delete/{id}")
    public ApiResult<Long> deleteDatabaseSource(
            @ApiParam(value = "要删除的数据库源ID", required = true)
            @PathVariable("id") @NotNull Long id) {
        return sysPlatformDatabaseSourceService.delete(id);
    }

    @ApiOperation("测试数据库连接配置的可用性")
    @PostMapping("/testConnection")
    public ApiResult<String> testDatabaseConnection(
            @ApiParam(value = "包含数据库连接配置信息的对象", required = true)
            @Valid @RequestBody DatabaseConfigDTO config) {
        return sysPlatformDatabaseSourceService.testConnection(config);
    }
    @ApiOperation("测试数据库连接配置的可用性")
    @PostMapping("/testConnectionById/{dsId}")
    public ApiResult<String> testDatabaseConnection(@PathVariable("dsId") @NotNull Long dsId) {
        return sysPlatformDatabaseSourceService.testConnectionById(dsId);
    }
}