package com.elitescloud.cloudt.platform.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import cn.zhxu.bs.BeanSearcher;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.platform.convert.SysPlatformApiMangeConvert;
import com.elitescloud.cloudt.platform.convert.SysPlatformApiParameterConvert;
import com.elitescloud.cloudt.platform.model.constant.PlatformApiFieldTypeEnum;
import com.elitescloud.cloudt.platform.model.constant.PlatformApiRequestTypeEnum;
import com.elitescloud.cloudt.platform.model.entity.SysPlatformApiParameterDO;
import com.elitescloud.cloudt.platform.model.params.api.AddApiParam;
import com.elitescloud.cloudt.platform.model.params.api.ApiFieldParam;
import com.elitescloud.cloudt.platform.model.vo.sbean.SysApiManageQueryBean;
import com.elitescloud.cloudt.platform.service.SysPlatformApiManageService;
import com.elitescloud.cloudt.platform.service.repo.*;
import com.elitescloud.cloudt.system.vo.SysUdcVO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Service
@Slf4j
@RequiredArgsConstructor
public class SysPlatformApiManageServiceImpl implements SysPlatformApiManageService {

    private final SysPlatformApiManageRepo sysPlatformApiManageRepo;

    private final SysPlatformApiParameterRepo sysPlatformApiParameterRepo;


    private final SysPlatformMenusApiRepo sysPlatformMenusApiRepo;

    private final SysPlatformMenusApiRepoProc menusApiRepoProc;

    private final BeanSearcher beanSearcher;


    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Long> addApi(AddApiParam addApiParam) {
        var saveDo = SysPlatformApiMangeConvert.INSTANCE.saveParamToDo(addApiParam);
        //验证编码是否重复
        var list = sysPlatformApiManageRepo.findByApiCode(saveDo.getApiCode());
        if (!list.isEmpty()) {
//            var obj = list.get(0);
            return ApiResult.fail("接口编码重复");
        }
        //保存接口基本信息
        saveDo.setRequestType(addApiParam.getRequestType().name());
        sysPlatformApiManageRepo.save(saveDo);
        var apiId = saveDo.getId();
        //保存接口字段信息
        saveFields(apiId, addApiParam.getApiCode(), addApiParam.getInParams(), SysPlatformApiParameterDO.TYPE_IN);
        saveFields(apiId, addApiParam.getApiCode(), addApiParam.getOutParams(), SysPlatformApiParameterDO.TYPE_OUT);

        return ApiResult.ok(apiId);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Boolean> updateApi(Long id, AddApiParam updateApiParam) {
        sysPlatformApiManageRepo.findById(id).ifPresentOrElse(manageDO -> {
                    Assert.isTrue(CharSequenceUtil.equals(manageDO.getApiCode(), updateApiParam.getApiCode()), "接口编码不可修改");
                    Assert.isTrue(CharSequenceUtil.equals(manageDO.getAppCode(), updateApiParam.getAppCode()), "所属应用不可修改");
                    manageDO.setApiName(updateApiParam.getApiName());
                    manageDO.setApiPath(updateApiParam.getApiPath());
                    manageDO.setRequestType(updateApiParam.getRequestType().name());
                    manageDO.setApiDescribe(updateApiParam.getApiDescribe());
                    sysPlatformApiParameterRepo.deleteAllByApiId(id);
                    sysPlatformApiManageRepo.save(manageDO);
                    saveFields(id, updateApiParam.getApiCode(), updateApiParam.getInParams(), SysPlatformApiParameterDO.TYPE_IN);
                    saveFields(id, updateApiParam.getApiCode(), updateApiParam.getOutParams(), SysPlatformApiParameterDO.TYPE_OUT);
                },
                () -> {
                    throw new BusinessException("id不存在");
                });
        return ApiResult.ok(true);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Boolean> deleteApi(Long id) {
        var sysPlatformManageDo = sysPlatformApiManageRepo.findById(id);
        if (sysPlatformManageDo.isPresent()) {
            sysPlatformApiParameterRepo.deleteAllByApiId(id);
            sysPlatformApiManageRepo.deleteById(id);
            sysPlatformMenusApiRepo.deleteAllByApiId(id);
        } else {
            throw new BusinessException("id不存在");
        }
        return ApiResult.ok(true);
    }

    @Override
    public ApiResult<PagingVO<SysApiManageQueryBean>> query(Map<String, Object> param) {

        var obj = beanSearcher.search(SysApiManageQueryBean.class, param);

        var apiIds = obj.getDataList().stream().map(SysApiManageQueryBean::getId).collect(Collectors.toList());
        Map<Long, Long> apiMenuNumMap = apiIds.isEmpty() ? Collections.emptyMap() : menusApiRepoProc.countMenuOfApi(apiIds);

        var result = new ArrayList<SysApiManageQueryBean>();
        for (SysApiManageQueryBean s : obj.getDataList()) {
            s.setMenuNum(apiMenuNumMap.getOrDefault(s.getId(), 0L));
            s.setInParams(sysPlatformApiParameterRepo.findAllByApiIdAndInOutType(s.getId(), SysPlatformApiParameterDO.TYPE_IN)
                    .stream()
                    .map(SysPlatformApiParameterConvert.INSTANCE::doToVo)
                    .collect(Collectors.toList()));
            s.setOutParams(sysPlatformApiParameterRepo.findAllByApiIdAndInOutType(s.getId(), SysPlatformApiParameterDO.TYPE_OUT)
                    .stream()
                    .map(SysPlatformApiParameterConvert.INSTANCE::doToVo)
                    .collect(Collectors.toList()));
            result.add(s);
        }
        var page = new PagingVO<>(obj.getTotalCount().longValue(), result);
        return ApiResult.ok(page);
    }

    @Override
    public ApiResult<List<SysUdcVO>> getApiRequestTypeEnum() {
        return ApiResult.ok(
                List.of(
                        PlatformApiRequestTypeEnum.GET.getUdcVO(),
                        PlatformApiRequestTypeEnum.POST.getUdcVO(),
                        PlatformApiRequestTypeEnum.PUT.getUdcVO(),
                        PlatformApiRequestTypeEnum.DELETE.getUdcVO()
                )
        );

    }

    @Override
    public ApiResult<List<SysUdcVO>> getApiFieldTypeEnum() {
        return ApiResult.ok(
                List.of(
                        PlatformApiFieldTypeEnum.TEXT.getUdcVO(),
                        PlatformApiFieldTypeEnum.NUMBER.getUdcVO(),
                        PlatformApiFieldTypeEnum.DATE.getUdcVO()
                )
        );
    }

    private void saveFields(Long id, String code, List<ApiFieldParam> list, String inOutType) {
        if (!(list == null)) {
            for (ApiFieldParam field : list) {
                var sysPlatformApiParameterDO = new SysPlatformApiParameterDO();
                sysPlatformApiParameterDO = SysPlatformApiParameterConvert.INSTANCE.saveParamToDo(field);
                sysPlatformApiParameterDO.setApiId(id);
                sysPlatformApiParameterDO.setApiCode(code);
                sysPlatformApiParameterDO.setFieldType(field.getFieldType().name());
                sysPlatformApiParameterDO.setInOutType(inOutType);
                sysPlatformApiParameterRepo.save(sysPlatformApiParameterDO);
            }
        }
    }
}
