package com.elitescloud.cloudt.system.provider;

import com.elitescloud.boot.auth.model.OAuthToken;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.swagger.feignapi.DocumentedFeignApi;
import com.elitescloud.cloudt.Application;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.dto.SysRolePlainDTO;
import com.elitescloud.cloudt.system.dto.SysUserBasicDTO;
import com.elitescloud.cloudt.system.dto.req.UserRoleSaveDTO;
import com.elitescloud.cloudt.system.param.AuthUserIdQueryDTO;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.List;
import java.util.Set;

/**
 * 用户权限.
 *
 * @author Kaiser（wang shao）
 * @date 2/23/2023
 */
@FeignClient(name = Application.NAME, path = SysUserPermissionRpcService.URI)
@Validated
@DocumentedFeignApi( description = "用户权限服务")
public interface SysUserPermissionRpcService {
    String URI = Application.URI_PREFIX + "/userPermission";

    /**
     * 查询指定租户的所有角色信息
     *
     * @param tenantCode 租户编码，为空则获取当前租户
     * @return 角色列表
     */
    @GetMapping(value = "/allRoles")
    @DocumentedFeignApi( description = "查询指定租户的所有角色信息")
    ApiResult<List<IdCodeNameParam>> listAllRoles(@RequestParam(name = "tenantCode", required = false) String tenantCode);

    /**
     * 查询指定租户的所有角色信息
     *
     * @param tenantCode 租户编码，为空则获取当前租户
     * @return 角色列表
     */
    @GetMapping(value = "/allRolesInPlain")
    @DocumentedFeignApi( description = "查询指定租户的所有角色信息，带业务分类")
    ApiResult<List<SysRolePlainDTO>> listAllRolesInPlain(@RequestParam(name = "tenantCode", required = false) String tenantCode);

    /**
     * 查询指定租户组织的所有角色信息
     *
     * @param tenantOrgCode 租户组织编码，为空则获取当前租户组织
     * @return 角色列表
     */
    @GetMapping(value = "/allRoles/tenantOrg")
    @DocumentedFeignApi( description = "查询指定租户组织的所有角色信息")
    ApiResult<List<IdCodeNameParam>> listAllRolesByTenantOrg(@RequestParam(name = "tenantOrgCode", required = false) String tenantOrgCode);

    /**
     * 保存用户角色
     *
     * @param userId  用户ID
     * @param roleIds 角色ID
     * @return 用户ID
     */
    @DocumentedFeignApi( description = "保存用户角色")
    @PostMapping(value = "/saveUserRoles")
    ApiResult<Long> saveUserRoles(@NotNull(message = "用户ID为空") @RequestParam(name = "userId") Long userId,
                                  @RequestBody List<Long> roleIds);

    /**
     * 批量保存用户角色
     *
     * @param saveDTOList 保存信息
     * @return 用户ID
     */
    @PostMapping(value = "/saveUserRoles/batch")
    @DocumentedFeignApi( description = "批量保存用户角色")
    ApiResult<List<Long>> saveUserRoles(@RequestBody List<UserRoleSaveDTO> saveDTOList);

    /**
     * 根据角色编码查询用户ID
     *
     * @param roleCode 角色编码
     * @return 用户ID
     */
    @GetMapping(value = "/getUserIdByRoleCode")
    @DocumentedFeignApi( description = "根据角色编码查询用户ID")
    ApiResult<Set<Long>> getUserIdByRoleCode(@NotBlank(message = "角色编码为空") @RequestParam("roleCode") String roleCode);

    /**
     * 根据角色查询用户信息
     *
     * @param roleCode 角色编码
     * @return 用户列表
     */
    @GetMapping(value = "/listUserByRole")
    @DocumentedFeignApi( description = "根据角色查询用户信息")
    ApiResult<List<SysUserBasicDTO>> listUserByRole(@NotBlank(message = "角色编码为空") @RequestParam("roleCode") String roleCode);

    /**
     * 根据用户ID进行认证授权
     *
     * @param queryDTO 参数
     * @return token
     */
    @PostMapping(value = "/authUserByUserId")
    @DocumentedFeignApi( description = "根据用户ID进行认证授权")
    ApiResult<OAuthToken> authByUserId(@RequestBody @Valid AuthUserIdQueryDTO queryDTO);

    /**
     * 根据用户手机号进行认证授权
     *
     * @param queryDTO 参数
     * @return token
     */
    @PostMapping(value = "/authByUserMobile")
    @DocumentedFeignApi( description = "根据用户手机号进行认证授权")
    ApiResult<OAuthToken> authByUserMobile(@RequestBody @Valid AuthUserIdQueryDTO queryDTO);

    /**
     * 根据用户邮箱进行认证授权
     *
     * @param queryDTO 参数
     * @return token
     */
    @PostMapping(value = "/authByUserEmail")
    @DocumentedFeignApi( description = "根据用户邮箱进行认证授权")
    ApiResult<OAuthToken> authByUserEmail(@RequestBody @Valid AuthUserIdQueryDTO queryDTO);

    /**
     * 根据外部系统标识进行认证授权
     *
     * @param queryDTO 参数
     * @return token
     */
    @PostMapping(value = "/authByOuterKey")
    @DocumentedFeignApi( description = "根据用户邮箱进行认证授权")
    ApiResult<OAuthToken> authByOuterKey(@RequestBody @Valid AuthUserIdQueryDTO queryDTO);

    /**
     * ticket换token
     *
     * @param ticket ticket值
     * @return token（不含token类型Bearer）
     */
    @GetMapping(value = "/getTokenByTicket")
    @DocumentedFeignApi( description = "ticket换token")
    ApiResult<String> getTokenByTicket(@NotBlank(message = "ticket为空") @RequestParam("ticket") String ticket);
}
