package com.elitescloud.cloudt.system.provider.org;

import com.elitescloud.boot.common.param.IdCodeNameCheckParam;
import com.elitescloud.boot.swagger.feignapi.DocumentedFeignApi;
import com.elitescloud.cloudt.Application;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.dto.SysEmployeeBasicDTO;
import com.elitescloud.cloudt.system.dto.SysEmployeeDetailDTO;
import com.elitescloud.cloudt.system.dto.req.EmployeePageQueryDTO;
import com.elitescloud.cloudt.system.dto.req.EmployeeQueryDTO;
import com.elitescloud.cloudt.system.dto.req.EmployeeUpsertDTO;
import com.elitescloud.cloudt.system.dto.resp.EmployeePageRespDTO;
import com.elitescloud.cloudt.system.dto.resp.EmployeeUnderlingDTO;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.List;
import java.util.Set;

/**
 * 员工相关服务.
 *
 * @author Kaiser（wang shao）
 * @date 2022/12/20
 */
@FeignClient(name = Application.NAME, path = EmployeeRpcService.URI)
@Validated
@DocumentedFeignApi( description = "员工相关服务")
public interface EmployeeRpcService {
    String URI = Application.URI_PREFIX + "/employee";

    /**
     * 员工创建与修改
     * <p>
     * 根据id或code确定员工唯一性，如果存在则修改，否则新增
     *
     * @param upsertDTO 员工信息
     * @return 员工编号
     */
    @PostMapping(value = "/upsert")
    @DocumentedFeignApi( description = "员工创建与修改:根据id或code确定员工唯一性，如果存在则修改，否则新增")
    ApiResult<String> upsert(@RequestBody @Valid EmployeeUpsertDTO upsertDTO);

    /**
     * 批量保存员工信息
     *
     * @param upsertDTOList 员工信息列表
     * @return 校验结果
     */
    @PostMapping(value = "/upsertBatch")
    @DocumentedFeignApi( description = "批量保存员工信息")
    ApiResult<List<String>> upsertBatch(@RequestBody @NotEmpty(message = "员工信息为空") @Valid List<EmployeeUpsertDTO> upsertDTOList);

    /**
     * 检查员工信息
     *
     * @param upsertDTOList 员工信息列表
     * @return 校验结果
     */
    @PostMapping(value = "/checkBatch")
    @DocumentedFeignApi( description = "检查员工信息")
    ApiResult<List<String>> checkBatch(@RequestBody @NotEmpty(message = "员工信息为空") @Valid List<EmployeeUpsertDTO> upsertDTOList);

    /**
     * 根据员工编号修改员工状态
     *
     * @param code    员工编号
     * @param enabled 启用状态
     * @return 员工编号
     */
    @PatchMapping(value = "/updateEnabledByCode")
    @DocumentedFeignApi( description = "根据员工编号修改员工状态")
    ApiResult<String> updateEnabledByCode(@NotBlank(message = "员工编号为空") @RequestParam("code") String code,
                                          @NotNull(message = "启用状态为空") @RequestParam("enabled") Boolean enabled);

    /**
     * 更新手机号
     *
     * @param id         用户ID
     * @param mobile     手机号
     * @param updateUser 是否更新账号里的手机号
     * @return 用户ID
     */
    @PutMapping(value = "/update/mobile")
    @DocumentedFeignApi( description = "更新手机号")
    ApiResult<Long> updateMobile(@NotNull(message = "用户ID为空") @RequestParam("id") Long id,
                                 @RequestParam(name = "mobile", required = false) String mobile,
                                 @RequestParam(name = "updateUser", defaultValue = "false") Boolean updateUser);

    /**
     * 更新邮箱
     *
     * @param id         用户ID
     * @param email      邮箱
     * @param updateUser 是否更新账号里的邮箱
     * @return 用户ID
     */
    @PutMapping(value = "/update/email")
    @DocumentedFeignApi( description = "更新邮箱")
    ApiResult<Long> updateEmail(@NotNull(message = "用户ID为空") @RequestParam("id") Long id,
                                @RequestParam(name = "email", required = false) String email,
                                @RequestParam(name = "updateUser", defaultValue = "false") Boolean updateUser);

    /**
     * 根据员工ID获取员工信息
     *
     * @param id 员工ID
     * @return 员工信息
     */
    @GetMapping(value = "/getById")
    @DocumentedFeignApi( description = "根据员工ID获取员工信息")
    ApiResult<SysEmployeeBasicDTO> getById(@NotNull(message = "员工ID为空") @RequestParam(name = "id") Long id);

    /**
     * 根据员工ID获取员工号
     *
     * @param id 员工ID
     * @return 员工号
     */
    @GetMapping(value = "/getCode")
    @DocumentedFeignApi( description = "根据员工ID获取员工号")
    ApiResult<String> getCode(@NotNull(message = "员工ID为空") @RequestParam(name = "id") Long id);

    /**
     * 根据员工ID获取账号ID
     *
     * @param id 员工ID
     * @return 账号ID
     */
    @GetMapping(value = "/getUserId")
    @DocumentedFeignApi( description = "根据员工ID获取账号ID")
    ApiResult<Long> getUserId(@NotNull(message = "员工ID为空") @RequestParam(name = "id") Long id);

    /**
     * 根据员工ID获取登录号
     *
     * @param id 员工ID
     * @return 登录号
     */
    @GetMapping(value = "/getUsername")
    @DocumentedFeignApi( description = "根据员工ID获取登录号")
    ApiResult<String> getUsername(@NotNull(message = "员工ID为空") @RequestParam(name = "id") Long id);

    /**
     * 根据员工ID获取员工信息
     *
     * @param code 员工编码
     * @return 员工信息
     */
    @GetMapping(value = "/getByCode")
    @DocumentedFeignApi( description = "根据员工ID获取员工信息")
    ApiResult<SysEmployeeBasicDTO> getByCode(@NotBlank(message = "员工编号为空") @RequestParam(name = "code") String code);

    /**
     * 根据账号获取员工信息
     *
     * @param username 登录账号
     * @return 员工信息
     */
    @GetMapping(value = "/getByUsername")
    @DocumentedFeignApi( description = "根据账号获取员工信息")
    ApiResult<SysEmployeeBasicDTO> getByUsername(@NotBlank(message = "用户账号为空") @RequestParam(name = "username") String username);

    /**
     * 根据用户ID获取所属员工
     *
     * @param userId 用户ID
     * @return 员工信息
     */
    @GetMapping(value = "/getByUserId")
    @DocumentedFeignApi( description = "根据用户ID获取所属员工")
    ApiResult<SysEmployeeBasicDTO> getByUserId(@NotNull(message = "用户ID为空") @RequestParam(name = "id") Long userId);

    /**
     * 根据员工ID获取详细信息
     *
     * @param id 员工ID
     * @return 详细信息
     */
    @GetMapping(value = "/getDetailById")
    @DocumentedFeignApi( description = "根据员工ID获取详细信息")
    ApiResult<SysEmployeeDetailDTO> getDetailById(@NotNull(message = "员工ID为空") @RequestParam(name = "id") Long id);

    /**
     * 根据员工编号获取详细信息
     *
     * @param code 员工编号
     * @return 详细信息
     */
    @GetMapping(value = "/getDetailByCode")
    @DocumentedFeignApi( description = "根据员工编号获取详细信息")
    ApiResult<SysEmployeeDetailDTO> getDetailByCode(@NotBlank(message = "员工编号为空") @RequestParam(name = "code") String code);

    /**
     * 根据用户ID获取详细信息
     *
     * @param userId 账号ID
     * @return 详细信息
     */
    @GetMapping(value = "/getDetailByUserId")
    @DocumentedFeignApi( description = "根据用户ID获取详细信息")
    ApiResult<SysEmployeeDetailDTO> getDetailByUserId(@NotNull(message = "用户ID为空") @RequestParam(name = "userId") Long userId);

    /**
     * 根据员工账号获取详细信息
     *
     * @param username 登录张阿红
     * @return 详细信息
     */
    @GetMapping(value = "/getDetailByUsername")
    @DocumentedFeignApi( description = "根据员工账号获取详细信息")
    ApiResult<SysEmployeeDetailDTO> getDetailByUsername(@NotBlank(message = "登录账号为空") @RequestParam(name = "username") String username);

    /**
     * 根据员工编号获取详细信息
     *
     * @param codes 员工编号
     * @return 详细信息
     */
    @PostMapping(value = "/getDetailListByCode")
    @DocumentedFeignApi( description = "根据员工编号获取详细信息")
    ApiResult<List<SysEmployeeDetailDTO>> getDetailListByCode(@RequestBody @NotEmpty(message = "员工编号为空") Set<String> codes);

    /**
     * 获取下属员工
     *
     * @param code    员工编号
     * @param cascade 是否级联获取所有下属员工，默认否
     * @param enabled 员工状态
     * @return 员工列表
     */
    @GetMapping(value = "/getUnderlingByCode")
    @DocumentedFeignApi( description = "获取下属员工")
    ApiResult<List<EmployeeUnderlingDTO>> getUnderlingByCode(@NotBlank(message = "员工编号为空") @RequestParam(name = "code") String code,
                                                             @RequestParam(name = "cascade", required = false, defaultValue = "false") Boolean cascade,
                                                             @RequestParam(name = "enabled", required = false) Boolean enabled);

    /**
     * 查询员工列表
     *
     * @param queryDTO 查询参数
     * @return 员工列表
     */
    @PostMapping(value = "/queryList")
    @DocumentedFeignApi( description = "查询员工列表")
    ApiResult<List<SysEmployeeBasicDTO>> queryList(@RequestBody EmployeeQueryDTO queryDTO);

    /**
     * 分页查询员工
     *
     * @param queryDTO 查询参数
     * @return 员工列表
     */
    @PostMapping(value = "/queryByPage")
    @DocumentedFeignApi( description = "分页查询员工")
    ApiResult<PagingVO<EmployeePageRespDTO>> queryByPage(@RequestBody EmployeePageQueryDTO queryDTO);

    /**
     * 绑定员工的业务员
     *
     * @param id         员工ID
     * @param salesmanId 业务员ID，为空则是解绑
     * @return 绑定结果
     */
    @PatchMapping(value = "/bind/salesman")
    @DocumentedFeignApi( description = "绑定员工的业务员")
    ApiResult<Boolean> bindSalesman(@NotNull(message = "员工ID为空") @RequestParam(name = "id") Long id,
                                    @RequestParam(name = "salesmanId", required = false) String salesmanId);

    /**
     * 获取员工的业务员ID
     *
     * @param id 员工ID
     * @return 业务员ID
     */
    @GetMapping(value = "/getSalesmanId")
    @DocumentedFeignApi( description = "获取员工的业务员ID")
    ApiResult<List<String>> getSalesmanId(@NotNull(message = "员工ID为空") @RequestParam(name = "id") Long id);

    /**
     * 获取员工的业务员ID
     *
     * @param userId 账号ID
     * @return 业务员ID
     */
    @GetMapping(value = "/getSalesmanIdByUserId")
    @DocumentedFeignApi( description = "获取员工的业务员ID")
    ApiResult<List<String>> getSalesmanIdByUserId(@NotNull(message = "账号ID为空") @RequestParam(name = "userId") Long userId);


    /**
     * 获取员工用户ID对应的角色列表
     *
     * @param userId 账号ID
     * @return 角色列表
     */
    @GetMapping(value = "/listRolesByUserId")
    @DocumentedFeignApi( description = "获取员工用户ID对应的角色列表")
    ApiResult<List<IdCodeNameCheckParam>> listRolesByUserId(@NotNull(message = "账号ID为空") @RequestParam(name = "userId") Long userId);
}
