package com.elitescloud.cloudt.system.seq.service.repo;

import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.system.constant.SysNumType;
import com.elitescloud.cloudt.system.seq.model.bo.SysSeqRuleDtlBO;
import com.elitescloud.cloudt.system.service.model.entity.QSysPlatformNumberRuleDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysPlatformNumberRuleDtlDO;
import com.elitescloud.cloudt.system.service.model.entity.SysPlatformNumberRuleDtlDO;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import com.querydsl.jpa.JPAExpressions;
import org.springframework.stereotype.Repository;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/11/14
 */
@Repository
public class SeqRuleDtlRepoProc extends BaseRepoProc<SysPlatformNumberRuleDtlDO> {
    private static final QSysPlatformNumberRuleDtlDO QDO = QSysPlatformNumberRuleDtlDO.sysPlatformNumberRuleDtlDO;
    private static final QSysPlatformNumberRuleDO QDO_RULE = QSysPlatformNumberRuleDO.sysPlatformNumberRuleDO;

    public SeqRuleDtlRepoProc() {
        super(QDO);
    }

    /**
     * 根据规则ID删除
     *
     * @param ruleId
     */
    public void deleteByRuleId(@NotNull Long ruleId) {
        super.deleteByValue(QDO.ruleId, ruleId);
    }

    public void deleteByRuleCode(@NotBlank String appCode, @NotBlank String ruleCode) {
        super.jpaQueryFactory.delete(QDO)
                .where(QDO.ruleId.eq(
                                JPAExpressions.select(QDO_RULE.id)
                                        .from(QDO_RULE)
                                        .where(QDO_RULE.ruleCode.eq(ruleCode).and(QDO_RULE.appCode.eq(appCode)))
                        )
                );
    }

    /**
     * 判断下一编号是否在用
     *
     * @param nextNumId
     * @return
     */
    public boolean isInUseForNextNum(@NotNull Long nextNumId) {
        var predicate = QDO.numberPattern.eq(nextNumId.toString())
                .and(QDO.numberType.eq(SysNumType.NN.name()));
        return jpaQueryFactory.select(QDO.id)
                .from(QDO)
                .where(predicate)
                .limit(1)
                .fetchOne() != null;
    }

    /**
     * 判断下一编号是否在用
     *
     * @param nextNumCode
     * @return
     */
    public boolean isInUseForNextNum(@NotBlank String appCode, @NotBlank String nextNumCode) {
        var predicate = QDO.numberPattern.eq(nextNumCode).and(QDO.appCode.eq(appCode))
                .and(QDO.numberType.eq(SysNumType.NN.name()));
        return jpaQueryFactory.select(QDO.id)
                .from(QDO)
                .where(predicate)
                .limit(1)
                .fetchOne() != null;
    }

    /**
     * 根据规则ID查询明细
     *
     * @param ruleId
     * @return
     */
    public List<SysPlatformNumberRuleDtlDO> queryByRuleId(@NotNull Long ruleId) {
        return jpaQueryFactory.select(QDO)
                .from(QDO)
                .where(QDO.ruleId.eq(ruleId))
                .orderBy(QDO.seq.asc())
                .fetch();
    }

    /**
     * 根据规则ID查询明细
     *
     * @param ruleId
     * @return
     */
    public List<SysSeqRuleDtlBO> queryBoByRuleId(@NotNull Long ruleId) {
        return jpaQueryFactory.select(qBeanBo())
                .from(QDO)
                .where(QDO.ruleId.eq(ruleId))
                .orderBy(QDO.seq.asc())
                .fetch();
    }

    /**
     * 根据规则查询
     *
     * @param ruleIds
     * @return
     */
    public Map<Long, List<SysSeqRuleDtlBO>> listBo(@NotEmpty Collection<Long> ruleIds) {
        return jpaQueryFactory.select(qBeanBo())
                .from(QDO)
                .where(QDO.ruleId.in(ruleIds))
                .stream()
                .collect(Collectors.groupingBy(SysSeqRuleDtlBO::getRuleId));
    }

    private QBean<SysSeqRuleDtlBO> qBeanBo() {
        return Projections.bean(SysSeqRuleDtlBO.class, QDO.id, QDO.appCode, QDO.ruleId, QDO.seq,
                QDO.numberType, QDO.numberPattern, QDO.nnLen, QDO.remark);
    }
}
