package com.elitescloud.cloudt.system.quartz.service.impl;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.util.DatetimeUtil;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.dto.resp.SysQuartzJobRespDTO;
import com.elitescloud.cloudt.system.quartz.common.SysQuartzTriggerType;
import com.elitescloud.cloudt.system.quartz.model.bo.schedule.*;
import com.elitescloud.cloudt.system.quartz.service.SysQuartzQueryService;
import com.elitescloud.cloudt.system.quartz.service.repo.SysQuartzJobRepoProc;
import com.fasterxml.jackson.core.type.TypeReference;
import org.quartz.CronExpression;
import org.quartz.CronScheduleBuilder;
import org.quartz.spi.MutableTrigger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.text.ParseException;
import java.util.*;
import java.util.function.Predicate;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2025/10/9 周四
 */
@Component()
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
public class SysQuartzQueryServiceImpl implements SysQuartzQueryService {
    private static final Logger logger = LoggerFactory.getLogger(SysQuartzQueryServiceImpl.class);

    @Autowired
    private SysQuartzJobRepoProc sysQuartzJobRepoProc;

    @Override
    public ApiResult<SysQuartzJobRespDTO> getJob(String jobCode) {
        var jobDO = sysQuartzJobRepoProc.getByJobCode(jobCode);
        if (jobDO == null) {
            return ApiResult.ok(null);
        }

        SysQuartzJobRespDTO respDTO = new SysQuartzJobRespDTO();
        respDTO.setJobCode(jobDO.getJobCode());
        respDTO.setGroupCode(jobDO.getGroupCode());
        respDTO.setJobName(jobDO.getJobName());
        respDTO.setEnabled(jobDO.getEnabled());
        respDTO.setDescription(jobDO.getDescription());
        respDTO.setJobData(JSONUtil.json2Obj(jobDO.getJobDataJson(), new TypeReference<HashMap<String, String>>() {
        }));
        respDTO.setTriggerType(jobDO.getTriggerType());
        respDTO.setTriggerJson(jobDO.getTriggerJson());
        respDTO.setJobClass(jobDO.getJobClass());
        respDTO.setStarDate(jobDO.getStarDate());
        respDTO.setEndDate(jobDO.getEndDate());
        return ApiResult.ok(respDTO);
    }

    @Override
    public ApiResult<List<String>> getNextFiredTime(String jobCode, Integer count) {
        // 查询任务配置
        var jobDO = sysQuartzJobRepoProc.getByJobCode(jobCode);
        if (jobDO == null) {
            return ApiResult.ok(Collections.emptyList());
        }

        // 转换触发器
        var sysQuartzTrigger = convertTrigger(jobDO.getTriggerType(), jobDO.getTriggerJson());
        if (sysQuartzTrigger == null) {
            return ApiResult.ok(Collections.emptyList());
        }

        if (count == null || count < 0) {
            count = 6;
        }
        Predicate<Date> validator = null;
        List<String> triggerTimes = new ArrayList<>(count);
        var trigger = (MutableTrigger) sysQuartzTrigger.buildTrigger();
        Date afterTime = new Date();
        while (triggerTimes.size() < count) {
            afterTime = trigger.getFireTimeAfter(afterTime);

            String triggerTime = DatetimeUtil.toStr(afterTime);
            if (validator == null || validator.test(afterTime)) {
                triggerTimes.add(triggerTime);
                continue;
            }
            logger.info("下一次触发时间：{}，不满足条件，跳过", triggerTime);
        }

        return ApiResult.ok(triggerTimes);
    }

    private SysQuartzTrigger convertTrigger(String triggerType, String triggerJson) {
        if (StrUtil.isBlank(triggerType) || StrUtil.isBlank(triggerJson)) {
            return null;
        }

        SysQuartzTriggerType triggerTypeEnum = SysQuartzTriggerType.getByValue(triggerType);
        Assert.notNull(triggerTypeEnum, "触发器类型错误");

        return switch (triggerTypeEnum) {
            case CRON -> JSONUtil.json2Obj(triggerJson, new TypeReference<CronSysQuartzTrigger>() {
            });
            case INTERVAL_SIMPLE -> JSONUtil.json2Obj(triggerJson, new TypeReference<IntervalSimpleSysQuartzTrigger>() {
            });
            case INTERVAL_CALENDAR ->
                    JSONUtil.json2Obj(triggerJson, new TypeReference<IntervalCalendarSysQuartzTrigger>() {
                    });
            case INTERVAL_DAILY_TIME -> JSONUtil.json2Obj(triggerJson, new TypeReference<DailyTimeSysQuartzTrigger>() {
            });
            default -> throw new IllegalArgumentException("不支持的触发器类型");
        };
    }
}
