package com.elitescloud.cloudt.system.quartz.model.bo.schedule;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import com.elitescloud.boot.util.cron.constant.CronWeek;
import com.elitescloud.cloudt.system.quartz.common.SysQuartzTriggerType;
import org.quartz.*;
import org.quartz.impl.triggers.DailyTimeIntervalTriggerImpl;

import java.io.Serial;
import java.time.LocalTime;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 基于时间的.
 *
 * @author Kaiser（wang shao）
 * @date 2025/10/11 周六
 */
public class DailyTimeSysQuartzTrigger implements SysQuartzTrigger {
    @Serial
    private static final long serialVersionUID = 4096913218789438545L;

    /**
     * 时间间隔
     */
    private Integer interval;
    /**
     * 时间间隔单位
     */
    private IntervalUnit intervalUnit;

    /**
     * 执行的时间
     */
    private Set<CronWeek> dayOfWeeks;

    /**
     * 开始时间
     */
    private LocalTime startTime;

    /**
     * 结束时间
     */
    private LocalTime endTime;

    /**
     * 重复次数（每天）
     */
    private Integer repeatCountOfDay;

    /**
     * 重复次数（一共）
     */
    private Integer repeatCount;


    @Override
    public SysQuartzTriggerType triggerType() {
        return SysQuartzTriggerType.INTERVAL_DAILY_TIME;
    }

    @Override
    public void validate() {
        Assert.notNull(interval, "间隔时间不能为空");
        Assert.isTrue(interval > 0, "间隔时间不能小于0");
        Assert.notNull(intervalUnit, "间隔时间单位不能为空");

        Assert.isTrue(repeatCountOfDay == null || repeatCountOfDay > 0, "重复次数不能为0");
        Assert.isTrue(repeatCount == null || repeatCount != 0, "重复次数不能为0");
    }

    @Override
    public Trigger buildTrigger() {
        DailyTimeIntervalTriggerImpl trigger = new DailyTimeIntervalTriggerImpl();
        // 时间间隔
        DateBuilder.IntervalUnit unit = switch (intervalUnit) {
            case SECOND -> DateBuilder.IntervalUnit.SECOND;
            case MINUTE -> DateBuilder.IntervalUnit.MINUTE;
            case HOUR -> DateBuilder.IntervalUnit.HOUR;
        };
        trigger.setRepeatInterval(interval);
        trigger.setRepeatIntervalUnit(unit);

        // 星期
        if (CollUtil.isNotEmpty(dayOfWeeks)) {
            trigger.setDaysOfWeek(dayOfWeeks.stream().map(CronWeek::getValue).collect(Collectors.toSet()));
        }

        // 时间
        if (startTime != null) {
            trigger.setStartTimeOfDay(TimeOfDay.hourMinuteAndSecondOfDay(startTime.getHour(), startTime.getMinute(), startTime.getSecond()));
        }
        if (endTime != null) {
            trigger.setEndTimeOfDay(TimeOfDay.hourMinuteAndSecondOfDay(endTime.getHour(), endTime.getMinute(), endTime.getSecond()));
        }

        // 重复次数
        if (repeatCount != null) {
            trigger.setRepeatCount(repeatCount);
        }
        return trigger;
    }

    @Override
    public ScheduleBuilder<?> buildScheduleBuilder() {
        DailyTimeIntervalScheduleBuilder scheduleBuilder = (DailyTimeIntervalScheduleBuilder) buildTrigger().getScheduleBuilder();
        if (repeatCountOfDay != null) {
            scheduleBuilder.endingDailyAfterCount(repeatCountOfDay);
        }

        return scheduleBuilder;
    }

    public Integer getInterval() {
        return interval;
    }

    public void setInterval(Integer interval) {
        this.interval = interval;
    }

    public IntervalUnit getIntervalUnit() {
        return intervalUnit;
    }

    public void setIntervalUnit(IntervalUnit intervalUnit) {
        this.intervalUnit = intervalUnit;
    }

    public Set<CronWeek> getDayOfWeeks() {
        return dayOfWeeks;
    }

    public void setDayOfWeeks(Set<CronWeek> dayOfWeeks) {
        this.dayOfWeeks = dayOfWeeks;
    }

    public LocalTime getStartTime() {
        return startTime;
    }

    public void setStartTime(LocalTime startTime) {
        this.startTime = startTime;
    }

    public LocalTime getEndTime() {
        return endTime;
    }

    public void setEndTime(LocalTime endTime) {
        this.endTime = endTime;
    }

    public Integer getRepeatCountOfDay() {
        return repeatCountOfDay;
    }

    public void setRepeatCountOfDay(Integer repeatCountOfDay) {
        this.repeatCountOfDay = repeatCountOfDay;
    }

    public Integer getRepeatCount() {
        return repeatCount;
    }

    public void setRepeatCount(Integer repeatCount) {
        this.repeatCount = repeatCount;
    }

    public enum IntervalUnit {SECOND, MINUTE, HOUR}
}
