package com.elitescloud.cloudt.system.seq.controller;

import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.common.annotation.BusinessObject;
import com.elitescloud.boot.common.annotation.BusinessObjectOperation;
import com.elitescloud.boot.common.annotation.businessobject.OperationTypeEnum;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.seq.model.vo.query.SeqRulePageQueryVO;
import com.elitescloud.cloudt.system.service.model.entity.SysPlatformNumberRuleDO;
import com.elitescloud.cloudt.system.constant.SysNumType;
import com.elitescloud.cloudt.system.seq.model.vo.resp.SeqRuleDetailsRespVO;
import com.elitescloud.cloudt.system.seq.model.vo.resp.SeqRulePageRespVO;
import com.elitescloud.cloudt.system.seq.model.vo.save.SeqRuleSaveVO;
import com.elitescloud.cloudt.system.seq.service.SeqRuleMngService;
import com.elitescloud.cloudt.system.constant.BusinessObjectConstant;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 发号规则管理.
 *
 * @author Kaiser（wang shao）
 * 2022/9/19
 */
@Api(tags = "发号规则管理")
@RestController
@RequestMapping(value = "/mng/seq/rule", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
@BusinessObject(businessType = BusinessObjectConstant.SYS_SEQ_RULE, businessDoClass = SysPlatformNumberRuleDO.class)
public class SeqRuleMngController {

    private final SeqRuleMngService seqRuleMngService;

    public SeqRuleMngController(SeqRuleMngService seqRuleMngService) {
        this.seqRuleMngService = seqRuleMngService;
    }

    /**
     * 分页查询记录
     *
     * @param queryVO 查询参数
     * @return 分页数据
     */
    @ApiOperation(value = "分页查询记录")
    @ApiOperationSupport(order = 1)
    @PostMapping(value = "/page")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT_PAGE, operationDescription = "分页查询"
    )
    public ApiResult<PagingVO<SeqRulePageRespVO>> page(@RequestBody SeqRulePageQueryVO queryVO) {
        return seqRuleMngService.pageMng(queryVO);
    }

    /**
     * 切换启用禁用
     *
     * @param id 记录ID
     * @return 切换结果
     */
    @ApiOperation(value = "切换启用禁用")
    @ApiOperationSupport(order = 2)
    @ApiImplicitParam(name = "id", value = "记录ID", required = true, dataType = "int")
    @PatchMapping(value = "/switch/enabled/{id}")
    @BusinessObjectOperation(operationType = OperationTypeEnum.UPDATE, operationDescription = "切换启用禁用"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> switchEnabled(@PathVariable("id") Long id) {
        return seqRuleMngService.updateEnabled(id);
    }

    /**
     * 获取记录详情
     *
     * @param id 记录ID
     * @return 记录信息
     */
    @ApiOperation(value = "获取记录详情")
    @ApiOperationSupport(order = 3)
    @ApiImplicitParam(name = "id", value = "记录ID", required = true, dataType = "int")
    @GetMapping(value = "/get/{id}")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "获取记录详情"
    )
    public ApiResult<SeqRuleDetailsRespVO> get(@PathVariable("id") Long id) {
        return seqRuleMngService.get(id);
    }

    /**
     * 修改记录
     *
     * @param id     记录ID
     * @param saveVO 记录信息
     * @return 修改结果
     */
    @ApiOperation(value = "修改记录")
    @ApiOperationSupport(order = 6)
    @ApiImplicitParam(name = "id", value = "记录ID", required = true, dataType = "int")
    @PutMapping(value = "/update/{id}")
    @BusinessObjectOperation(operationType = OperationTypeEnum.UPDATE, operationDescription = "修改记录"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> update(@PathVariable("id") Long id, @RequestBody @Valid SeqRuleSaveVO saveVO) {
        return seqRuleMngService.update(id, saveVO);
    }

    /**
     * 生成示例代码
     *
     * @param id 记录ID
     * @return 记录信息
     */
    @ApiOperation(value = "生成示例代码")
    @ApiOperationSupport(order = 7)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "记录ID", required = true, dataType = "int"),
            @ApiImplicitParam(name = "runtimeValues", value = "运行时参数", paramType = "body")
    })
    @PostMapping(value = "/get/{id}/sample")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "生成示例代码"
    )
    public ApiResult<String> getSampleCode(@PathVariable("id") Long id, @RequestBody List<String> runtimeValues) {
        return seqRuleMngService.getSample(id, runtimeValues);
    }

    /**
     * 取号类型
     *
     * @return 取号类型列表
     */
    @ApiOperation(value = "取号类型")
    @ApiOperationSupport(order = 21)
    @GetMapping(value = "/numType")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "取号类型"
    )
    public ApiResult<List<CodeNameParam>> numType() {
        var values = Arrays.stream(SysNumType.values())
                .map(t -> new CodeNameParam(t.name(), t.getDescription()))
                .collect(Collectors.toList());
        return ApiResult.ok(values);
    }
}
