package com.elitescloud.cloudt.system.quartz.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.util.DatetimeUtil;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.boot.util.ObjUtil;
import com.elitescloud.boot.util.StrUtil;
import com.elitescloud.boot.util.cron.CronUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.quartz.common.SysQuartzConstant;
import com.elitescloud.cloudt.system.quartz.common.SysQuartzTriggerTimeValidator;
import com.elitescloud.cloudt.system.quartz.model.bo.SysQuartzJobSaveParam;
import com.elitescloud.cloudt.system.quartz.model.entity.SysQuartzJobDO;
import com.elitescloud.cloudt.system.quartz.service.SysQuartzMngService;
import com.elitescloud.cloudt.system.quartz.service.provider.SysQuartzJobDelegate;
import com.elitescloud.cloudt.system.quartz.service.repo.SysQuartzJobRepoProc;
import org.quartz.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.*;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2025/9/20 周六
 */
@Component()
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
public class SysQuartzMngServiceImpl implements SysQuartzMngService {
    private static final Logger logger = LoggerFactory.getLogger(SysQuartzMngServiceImpl.class);

    private static final String TRIGGER_IDENTIFIER_SUFFIX = "_trigger";

    @Autowired
    private ObjectProvider<SysQuartzTriggerTimeValidator> triggerTimeValidators;
    @Autowired
    private Scheduler scheduler;
    @Autowired
    private SysQuartzJobRepoProc sysQuartzJobRepoProc;

    @Override
    public ApiResult<Boolean> started() {
        var started = isSchedulerStared();
        return ApiResult.ok(started);
    }

    @Override
    public ApiResult<Boolean> validateCron(String cron) {
        var validated = CronUtil.validateCronExpression(cron);
        return ApiResult.ok(validated);
    }

    @Override
    public ApiResult<List<String>> getNextFiredTime(String cron, Integer count) {
        if (count == null) {
            count = 6;
        }

        var nextFiredTime = CronUtil.getNextTriggerTimes(cron, count, time -> validateTriggerTime(DatetimeUtil.date2LocalDateTime(time)) == null);
        if (nextFiredTime.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        return ApiResult.ok(nextFiredTime.stream().map(DatetimeUtil::toStr).toList());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> saveJob(SysQuartzJobSaveParam saveParam) {
        // 检查参数
        checkAndFillDefault(saveParam);

        // 保存任务信息
        saveJobInfo(saveParam);

        // 注册调度任务
        try {
            registerJob(saveParam);
        } catch (Exception e) {
            logger.error("注册任务异常", e);
            throw new BusinessException("注册任务异常, 请联系管理员", e);
        }

        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> pauseJob(String jobCode) {
        var jobKey = getJobKey(jobCode);
        if (jobKey == null) {
            return ApiResult.fail("任务不存在");
        }

        try {
            if (scheduler.checkExists(jobKey)) {
                scheduler.pauseJob(jobKey);
            }
        } catch (SchedulerException e) {
            throw new BusinessException("暂停任务异常, 请联系管理员", e);
        }
        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> resumeJob(String jobCode) {
        var jobKey = getJobKey(jobCode);
        if (jobKey == null) {
            return ApiResult.fail("任务不存在");
        }

        try {
            if (scheduler.checkExists(jobKey)) {
                scheduler.resumeJob(jobKey);
            } else {
                return ApiResult.fail("任务不存在");
            }
        } catch (SchedulerException e) {
            throw new BusinessException("恢复任务异常, 请联系管理员", e);
        }
        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> deleteJob(String jobCode) {
        var jobKey = getJobKey(jobCode);
        if (jobKey == null) {
            return ApiResult.fail("任务不存在");
        }

        try {
            if (scheduler.checkExists(jobKey)) {
                scheduler.deleteJob(jobKey);
            }
        } catch (SchedulerException e) {
            throw new BusinessException("删除任务异常, 请联系管理员", e);
        }
        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> runOnceJob(String jobCode) {
        var jobKey = getJobKey(jobCode);
        if (jobKey == null) {
            return ApiResult.fail("任务不存在");
        }

        if (!isSchedulerStared()) {
            return ApiResult.fail("调度器未启动");
        }

        try {
            if (scheduler.checkExists(jobKey)) {
                JobDataMap jobDataMap = new JobDataMap();
                jobDataMap.put(SysQuartzConstant.JOB_DATA_KEY_SOURCE_RUNONCE, true);
                scheduler.triggerJob(jobKey, jobDataMap);
            }
        } catch (SchedulerException e) {
            throw new BusinessException("运行任务异常, 请联系管理员", e);
        }
        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> interruptJob(String jobCode) {
        var jobKey = getJobKey(jobCode);
        if (jobKey == null) {
            return ApiResult.fail("任务不存在");
        }

        try {
            if (scheduler.checkExists(jobKey)) {
                scheduler.interrupt(jobKey);
            }
        } catch (SchedulerException e) {
            throw new BusinessException("中断任务异常, 请联系管理员", e);
        }
        return ApiResult.ok(true);
    }

    private void registerJob(SysQuartzJobSaveParam saveParam) throws Exception {
        var jobDetailBuilder = JobBuilder.newJob(SysQuartzJobDelegate.class)
                .withIdentity(saveParam.getJobCode(), saveParam.getJobType().groupCode())
                .withDescription(saveParam.getJobType().description())
                .storeDurably();
        if (CollUtil.isNotEmpty(saveParam.getJobData())) {
            for (Map.Entry<String, String> entry : saveParam.getJobData().entrySet()) {
                jobDetailBuilder.usingJobData(entry.getKey(), entry.getValue());
            }
        }
        JobDetail jobDetail = jobDetailBuilder.build();

        TriggerBuilder<?> triggerBuilder = TriggerBuilder.newTrigger()
                .withIdentity(saveParam.getJobCode() + TRIGGER_IDENTIFIER_SUFFIX, saveParam.getJobType().groupCode() + TRIGGER_IDENTIFIER_SUFFIX)
                .withSchedule(saveParam.getTrigger().buildScheduleBuilder())
                .startAt(saveParam.getStartTime())
                .endAt(saveParam.getEndTime());
        var trigger = triggerBuilder.build();

        scheduler.scheduleJob(jobDetail, Set.of(trigger), true);
    }

    private void saveJobInfo(SysQuartzJobSaveParam saveParam) {
        SysQuartzJobDO jobInfo = sysQuartzJobRepoProc.getByJobCode(saveParam.getJobCode());
        if (jobInfo == null) {
            jobInfo = new SysQuartzJobDO();
            jobInfo.setJobCode(saveParam.getJobCode());
            jobInfo.setGroupCode(saveParam.getJobType().groupCode());
            jobInfo.setEnabled(true);
            jobInfo.setSysTenantId(saveParam.getTenantId());
        } else {
            Assert.isTrue(jobInfo.getGroupCode().equals(saveParam.getJobType().groupCode()), "任务组编码不能修改");
            Assert.isTrue(ObjUtil.equals(saveParam.getTenantId(), jobInfo.getSysTenantId()), "任务编码已存在");
        }

        jobInfo.setJobName(saveParam.getJobName());
        jobInfo.setDescription(saveParam.getDescription());
        jobInfo.setJobClass(saveParam.getJobClass().getName());
        jobInfo.setJobDataJson(JSONUtil.toJsonString(saveParam.getJobData()));

        // 触发器
        jobInfo.setTriggerType(saveParam.getTrigger().triggerType().name());
        jobInfo.setTriggerJson(JSONUtil.toJsonString(saveParam.getTrigger()));

        jobInfo.setStarDate(DatetimeUtil.date2LocalDateTime(saveParam.getStartTime()));
        jobInfo.setEndDate(DatetimeUtil.date2LocalDateTime(saveParam.getEndTime()));
        sysQuartzJobRepoProc.save(jobInfo);
    }

    private void checkAndFillDefault(SysQuartzJobSaveParam saveParam) {
        Assert.isTrue(isSchedulerStared(), "调度器未启动");

        Assert.notBlank(saveParam.getJobCode(), "任务编码不能为空");
        Assert.notNull(saveParam.getJobType(), "任务类型不能为空");

        Assert.notNull(saveParam.getJobClass(), "任务执行器类不能为空");
        Assert.isTrue(Job.class.isAssignableFrom(saveParam.getJobClass()), "不支持的任务执行器");
        try {
            var methodConstructor = saveParam.getJobClass().getConstructor();
        } catch (NoSuchMethodException e) {
            throw new BusinessException("任务执行器类未定义无参构造函数");
        }

        Assert.notNull(saveParam.getTrigger(), "任务触发器不能为空");
        saveParam.getTrigger().validate();

        if (saveParam.getStartTime() == null) {
            saveParam.setStartTime(new Date());
        }
    }

    private String validateTriggerTime(LocalDateTime time) {
        if (time == null) {
            return "时间不能为空";
        }

        for (SysQuartzTriggerTimeValidator validator : triggerTimeValidators) {
            var result = validator.validate(time);
            if (result != null) {
                return result;
            }
        }
        return null;
    }

    private JobKey getJobKey(String jobCode) {
        if (StrUtil.isBlank(jobCode)) {
            return null;
        }

        String groupCode = sysQuartzJobRepoProc.getJobGroupCode(jobCode);
        if (StrUtil.isBlank(groupCode)) {
            return null;
        }

        return JobKey.jobKey(jobCode, groupCode);
    }

    private boolean isSchedulerStared() {
        try {
            return scheduler.isStarted();
        } catch (SchedulerException e) {
            throw new BusinessException("获取调度器状态异常", e);
        }
    }
}
