package com.elitescloud.cloudt.comm.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.cloudt.comm.consumer.dto.ComPaymentTermRpcDTO;
import com.elitescloud.cloudt.comm.consumer.param.ComPaymentTermRpcDtoParam;
import com.elitescloud.cloudt.comm.convert.ComPaymentTermConvert;
import com.elitescloud.cloudt.comm.entity.ComPaymentTermDO;
import com.elitescloud.cloudt.comm.entity.QComPaymentTermDO;
import com.elitescloud.cloudt.comm.repo.ComPaymentTermRepo;
import com.elitescloud.cloudt.comm.repo.ComPaymentTermRepoProc;
import com.elitescloud.cloudt.comm.vo.param.ComPaymentTermQueryParamVO;
import com.elitescloud.cloudt.comm.vo.resp.ComPaymentTermRespVO;
import com.elitescloud.cloudt.comm.vo.save.ComPaymentTermSaveVO;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.system.service.IComPaymentTermService;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.jpa.impl.JPAQuery;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * <p>
 * 功能说明
 * </p>
 *
 * @author shihao.ma
 * 2020/7/2
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class ComPaymentTermServiceImpl implements IComPaymentTermService {

    private final ComPaymentTermRepo comPaymentTermRepo;

    private final ComPaymentTermRepoProc comPaymentTermRepoProc;

    /**
     * 创建支付条款
     *
     * @param paymentTerm 支付条款对象
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long createOne(ComPaymentTermSaveVO paymentTerm) {
        paymentTerm.setId(null);
//        checkSaveExist(paymentTerm);
        List<ComPaymentTermSaveVO> comPaymentTermDOS = validDateProcessing(Collections.singletonList(paymentTerm));
        comPaymentTermRepo.saveAll(comPaymentTermDOS.stream().map(ComPaymentTermConvert.INSTANCE::saveVOToDO).collect(Collectors.toList()));
        return 0L;
    }

    /**
     * 批量创建支付条款
     *
     * @param list 支付条款对象集合
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public List<Long> createBatch(List<ComPaymentTermSaveVO> list) {
        list.forEach(i -> i.setId(null));
        List<ComPaymentTermSaveVO> comPaymentTermDOS = validDateProcessing(list);
        comPaymentTermRepo.saveAll(comPaymentTermDOS.stream().map(ComPaymentTermConvert.INSTANCE::saveVOToDO).collect(Collectors.toList()));
        return Collections.EMPTY_LIST;
    }

    /**
     * 更新支付条款
     *
     * @param paymentTerm 支付条款对象
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void update(ComPaymentTermSaveVO paymentTerm) {
        if (comPaymentTermRepo.existsById(paymentTerm.getId())) {
            List<ComPaymentTermSaveVO> comPaymentTermDOS = validDateProcessing(Collections.singletonList(paymentTerm));
            comPaymentTermRepo.saveAll(comPaymentTermDOS.stream().map(ComPaymentTermConvert.INSTANCE::saveVOToDO).collect(Collectors.toList()));
        } else {
            throw new BusinessException("数据不存在");
        }
    }

    /**
     * 删除支付条款
     *
     * @param id 支付条款ID
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void deleteOne(Long id) {
        if (comPaymentTermRepo.existsById(id)) {
            comPaymentTermRepo.deleteById(id);
        } else {
            throw new BusinessException("数据不存在");
        }
    }

    /**
     * 批量删除支付条款
     *
     * @param paymentTermIdList 支付条款ID集合
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void deleteBatch(List<Long> paymentTermIdList) {
        List<ComPaymentTermDO> delList = new ArrayList<>();
        paymentTermIdList.forEach(i -> delList.add(new ComPaymentTermDO() {{
            setId(i);
        }}));
        comPaymentTermRepo.deleteAll(delList);
    }

    /**
     * 更新支付条款删除标识
     *
     * @param id 支付条款ID
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void updateDeleteFlag(Long id) {
        QComPaymentTermDO paymentTermDO = QComPaymentTermDO.comPaymentTermDO;
        Predicate predicate = paymentTermDO.deleteFlag.isNull().or(paymentTermDO.deleteFlag.eq(0));
        predicate = ExpressionUtils.and(predicate, paymentTermDO.id.eq(id));
        Optional<ComPaymentTermDO> comPaymentTermDO = comPaymentTermRepo.findOne(predicate);
        // 更新删除标识
        comPaymentTermDO.ifPresent(c -> {
            c.setDeleteFlag(1);
            comPaymentTermRepo.save(c);
        });
    }

    /**
     * 检索支付条款
     *
     * @param queryParam 分类查询、分页和排序对象
     * @return 分页结果集
     */
    @Override
    public PagingVO<ComPaymentTermRespVO> search(ComPaymentTermQueryParamVO queryParam) {
        return comPaymentTermRepoProc.pageForMng(queryParam);
    }

    @Override
    public ComPaymentTermRespVO getCodeOne(String code) {
        return findCodeOne(code).orElse(null);
    }

    @Override
    public ComPaymentTermRespVO getIdOne(Long id) {
        return findIdOne(id).orElse(null);
    }

    /**
     * 根据关键字段，返回对应的支付条款
     *
     * @param paymentTermCode 支付条款code
     * @return 支付条款对象
     */
    @Override
    public Optional<ComPaymentTermRespVO> findCodeOne(String paymentTermCode) {
        QComPaymentTermDO paymentTerm = QComPaymentTermDO.comPaymentTermDO;
        JPAQuery<ComPaymentTermRespVO> jpaQuery = comPaymentTermRepoProc.select();
        jpaQuery.where(paymentTerm.ptCode.eq(paymentTermCode));
        return Optional.ofNullable(jpaQuery.fetchOne());
    }

    /**
     * 根据关键字段，返回对应的支付条款
     *
     * @param paymentTermCodes 支付条款codes
     * @return 支付条款对象
     */
    @Override
    public List<ComPaymentTermRespVO> findCodeBatch(List<String> paymentTermCodes) {
        if (paymentTermCodes == null || paymentTermCodes.isEmpty()) {
            return new ArrayList<>();
        }
        QComPaymentTermDO paymentTerm = QComPaymentTermDO.comPaymentTermDO;
        JPAQuery<ComPaymentTermRespVO> jpaQuery = comPaymentTermRepoProc.select();
        jpaQuery.where(paymentTerm.ptCode.in(paymentTermCodes));
        return jpaQuery.fetch();
    }

    /**
     * 根据关键字段，返回对应的支付条款
     *
     * @param paymentTermId 支付条款ID
     * @return 支付条款对象
     */
    @Override
    public Optional<ComPaymentTermRespVO> findIdOne(Long paymentTermId) {
        QComPaymentTermDO paymentTerm = QComPaymentTermDO.comPaymentTermDO;
        JPAQuery<ComPaymentTermRespVO> jpaQuery = comPaymentTermRepoProc.select();
        jpaQuery.where(paymentTerm.id.eq(paymentTermId));
        return Optional.ofNullable(jpaQuery.fetchOne());
    }

    /**
     * 根据Id集合，查询支付条款
     *
     * @param idList 支付条款的Id集合
     * @return 支付条款对象集合
     */
    @Override
    public List<ComPaymentTermRespVO> findIdBatch(List<Long> idList) {
        return comPaymentTermRepo.findAllById(idList).stream().map(paymentTermDO -> {
            ComPaymentTermRespVO comPaymentTermVO = new ComPaymentTermRespVO();
            BeanUtils.copyProperties(paymentTermDO, comPaymentTermVO);
            return comPaymentTermVO;
        }).collect(Collectors.toList());
    }

    @Override
    public List<ComPaymentTermRespVO> list() {
        return comPaymentTermRepo.findAll().stream().map(ComPaymentTermConvert.INSTANCE::doToVO)
                .collect(Collectors.toList());
    }

    @Override
    public List<ComPaymentTermRpcDTO> findPaymentTermRpcDtoByParam(ComPaymentTermRpcDtoParam param) {
        return comPaymentTermRepoProc.findRpcDtoByParam(param);
    }

    private List<ComPaymentTermSaveVO> validDateProcessing(List<ComPaymentTermSaveVO> paymentTermList) {
        List<ComPaymentTermSaveVO> result = Collections.synchronizedList(new ArrayList<>());
        // 区间范围
        paymentTermList.parallelStream().forEach(i -> {
            i.setValidFrom(LocalDateTime.of(i.getValidFrom().toLocalDate(), LocalTime.of(0, 0, 0)));
            i.setValidTo(LocalDateTime.of(i.getValidTo().toLocalDate(), LocalTime.of(23, 59, 59)));
        });

        // 数据校验
        paymentTermList.forEach(i -> {
            if (CharSequenceUtil.isBlank(i.getPtCode())) {
                throw new BusinessException("数据错误，支付条款不能为空");
            }
            if (!i.getValidFrom().isBefore(i.getValidTo())) {
                throw new BusinessException("数据错误，失效日期不能在生效日期之前");
            }
        });
        List<String> ptCodes = paymentTermList.stream().map(ComPaymentTermSaveVO::getPtCode).distinct().collect(Collectors.toList());

        QComPaymentTermDO paymentTerm = QComPaymentTermDO.comPaymentTermDO;
        List<ComPaymentTermRespVO> paymentTermVOList = comPaymentTermRepoProc.select().where(paymentTerm.ptCode.in(ptCodes)).fetch();

        if (CollectionUtils.isEmpty(paymentTermVOList)) {
            return paymentTermList;
        }
        for (ComPaymentTermSaveVO paymentTermDO : paymentTermList) {
            List<ComPaymentTermRespVO> paymentTermVOS = paymentTermVOList.stream()
                    .filter(i -> paymentTermDO.getPtCode().equals(i.getPtCode())).collect(Collectors.toList());

            if (CollectionUtils.isEmpty(paymentTermVOS)) {
                result.add(paymentTermDO);
            } else {
                List<ComPaymentTermSaveVO> validProcessList = getValidProcessList(paymentTermDO, paymentTermVOS);
                result.addAll(validProcessList);
            }
        }
        return result;
    }

    /**
     * 校验获取 符合有效期处理的DO数据集合
     *
     * @param newDO          需校验的新增DO
     * @param paymentTermVOS 与该新增DO 外键一致的所有匹配数据
     * @return 符合有效期处理的DO数据集合
     */
    private List<ComPaymentTermSaveVO> getValidProcessList(ComPaymentTermSaveVO newDO, List<ComPaymentTermRespVO> paymentTermVOS) {
        List<ComPaymentTermSaveVO> result = new ArrayList<>();
        for (ComPaymentTermRespVO oldVO : paymentTermVOS) {
            if (oldVO.getValidTo().isBefore(newDO.getValidFrom()) || oldVO.getValidFrom().isAfter(newDO.getValidTo())) {
                /**
                 *      ______(老区间)______
                 *                              ______(新区间)______
                 *                      or
                 *                              ______(新区间)______
                 *      ______(老区间)______
                 */
                continue;
            }
            if (oldVO.getValidFrom().isBefore(newDO.getValidFrom())) {
                if (oldVO.getValidTo().isBefore(newDO.getValidTo()) || oldVO.getValidTo().isEqual(newDO.getValidTo())) {
                    /**
                     *        ___________ (老区间)______
                     *              ______(新区间)_____________
                     *                      or
                     *        ___________(老区间)______
                     *              ______(新区间)_____
                     */
                    ComPaymentTermSaveVO oldPaymentTermDO = new ComPaymentTermSaveVO();
                    BeanUtil.copyProperties(oldVO, oldPaymentTermDO);
                    oldPaymentTermDO.setValidTo(LocalDateTime.of(LocalDate.from(newDO.getValidFrom().minusDays(1)), LocalTime.of(23, 59, 59)));
                    result.add(oldPaymentTermDO);
                    continue;
                }
                if (oldVO.getValidTo().isAfter(newDO.getValidTo())) {
                    /**
                     *        ___________(老区间)________________
                     *              ______(新区间)______
                     */
                    ComPaymentTermSaveVO beforeOldPaymentTerm = new ComPaymentTermSaveVO();
                    ComPaymentTermSaveVO afterOldPaymentTerm = new ComPaymentTermSaveVO();
                    ComPaymentTermSaveVO oldPaymentTerm = new ComPaymentTermSaveVO();
                    BeanUtil.copyProperties(oldVO, oldPaymentTerm);
                    BeanUtil.copyProperties(oldVO, beforeOldPaymentTerm);
                    BeanUtil.copyProperties(oldVO, afterOldPaymentTerm);
                    beforeOldPaymentTerm.setValidTo(LocalDateTime.of(LocalDate.from(newDO.getValidFrom().minusDays(1)), LocalTime.of(23, 59, 59)));
                    afterOldPaymentTerm.setValidFrom(LocalDateTime.of(LocalDate.from(newDO.getValidTo().plusDays(1)), LocalTime.MIN));
                    oldVO.setDeleteFlag(1);
                    beforeOldPaymentTerm.setId(null);
                    afterOldPaymentTerm.setId(null);
                    oldPaymentTerm.setDeleteFlag(1);
                    result.add(oldPaymentTerm);
                    result.add(beforeOldPaymentTerm);
                    result.add(afterOldPaymentTerm);
                    continue;
                }
            }
            if (oldVO.getValidFrom().isAfter(newDO.getValidFrom()) || oldVO.getValidFrom().isEqual(newDO.getValidFrom())) {
                if (oldVO.getValidTo().isBefore(newDO.getValidTo()) || oldVO.getValidTo().isEqual(newDO.getValidTo())) {
                    /**
                     *          ______(老区间)______
                     *      __________(新区间)_______________
                     *                   or
                     *          ______(老区间)______
                     *          ______(新区间)______
                     *                   or
                     *          ______(老区间)______
                     *      __________(新区间)______
                     *
                     */
                    ComPaymentTermSaveVO oldPaymentTerm = new ComPaymentTermSaveVO();
                    BeanUtil.copyProperties(oldVO, oldPaymentTerm);
                    oldPaymentTerm.setDeleteFlag(1);
                    result.add(oldPaymentTerm);
                }
                if (oldVO.getValidTo().isAfter(newDO.getValidTo())) {
                    /**
                     *          ______(老区间)___________
                     *          ______(新区间)______
                     *                   or
                     *          ______(老区间)___________
                     *      __________(新区间)______
                     */
                    ComPaymentTermSaveVO oldPaymentTerm = new ComPaymentTermSaveVO();
                    BeanUtil.copyProperties(oldVO, oldPaymentTerm);
                    oldPaymentTerm.setValidFrom(LocalDateTime.of(LocalDate.from(newDO.getValidTo().plusDays(1)), LocalTime.MIN));
                    result.add(oldPaymentTerm);
                }
            }

        }
        result.add(newDO); // 把本次操作的DO 添加至操作集合中
        return result;
    }
}
