package com.elitescloud.cloudt.log.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.log.common.LogLevel;
import com.elitescloud.cloudt.log.model.document.LogStashDocument;
import com.elitescloud.cloudt.log.model.vo.param.LogStashQueryParam;
import com.elitescloud.cloudt.log.model.vo.resp.LogStashRecordRespVO;
import com.elitescloud.cloudt.log.service.LogStashService;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.aggregations.Aggregation;
import org.elasticsearch.search.aggregations.bucket.composite.CompositeAggregation;
import org.elasticsearch.search.aggregations.bucket.composite.CompositeAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.composite.TermsValuesSourceBuilder;
import org.elasticsearch.search.fetch.subphase.highlight.HighlightBuilder;
import org.elasticsearch.search.sort.SortBuilders;
import org.elasticsearch.search.sort.SortOrder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.elasticsearch.core.ElasticsearchAggregations;
import org.springframework.data.elasticsearch.core.ElasticsearchRestTemplate;
import org.springframework.data.elasticsearch.core.SearchHits;
import org.springframework.data.elasticsearch.core.document.Document;
import org.springframework.data.elasticsearch.core.mapping.IndexCoordinates;
import org.springframework.data.elasticsearch.core.query.NativeSearchQueryBuilder;
import org.springframework.data.mapping.callback.EntityCallback;
import org.springframework.data.mapping.callback.EntityCallbacks;
import org.springframework.lang.NonNull;
import org.springframework.util.CollectionUtils;

import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2021/07/31
 */
public class LogStashServiceFilebeat implements LogStashService {
    private static final Logger logger = LoggerFactory.getLogger(LogStashServiceFilebeat.class);

    private final RestHighLevelClient client;
    private final String preIndex;

    public LogStashServiceFilebeat(RestHighLevelClient client, String preIndex) {
        this.client = client;
        this.preIndex = preIndex;
        init();
    }

    @Override
    public ApiResult<PagingVO<LogStashRecordRespVO>> search(LogStashQueryParam queryParam) {
        var searchHits = searchHits(queryParam);

        if (searchHits == null || !searchHits.hasSearchHits()) {
            PagingVO<LogStashRecordRespVO> pagingVO = PagingVO.<LogStashRecordRespVO>builder()
                    .total(searchHits == null ? 0 : searchHits.getTotalHits())
                    .records(Collections.emptyList())
                    .build();
            return ApiResult.ok(pagingVO);
        }

        var records = searchHits.get().map(t -> {
                    var document = t.getContent();
                    var vo = new LogStashRecordRespVO();
                    vo.setPreIndex(document.getPreIndex());
                    vo.setAppName(document.getAppName());
                    vo.setLogLevel(document.getLevel());
                    vo.setMsg(document.getMsg());
                    vo.setTime(LocalDateTime.parse(document.getTime(), FORMATTER));
                    vo.setMessage(document.getMessage());
                    return vo;
                }).sorted(Comparator.comparing(LogStashRecordRespVO::getTime).reversed())
                .collect(Collectors.toList());
        return ApiResult.ok(PagingVO.<LogStashRecordRespVO>builder().total(searchHits.getTotalHits()).records(records).build());
    }

    @Override
    public ApiResult<List<String>> logLevelList() {
        List<String> levelNames = Arrays.stream(LogLevel.values()).map(LogLevel::name).collect(Collectors.toList());
        return ApiResult.ok(levelNames);
    }

    @Override
    public ApiResult<List<String>> appNameList(Integer size, LocalDateTime start, LocalDateTime end) {
        IndexCoordinates indexCoordinates = null;
        try {
            indexCoordinates = obtainIndexCoordinates(start, end);
        } catch (Exception e) {
            throw new BusinessException("查询失败", e);
        }
        if (indexCoordinates == null) {
            return ApiResult.ok(Collections.emptyList());
        }

        var appNameList = searchAppName(indexCoordinates, size);
        return ApiResult.ok(appNameList);
    }

    private List<String> searchAppName(IndexCoordinates indexCoordinates, Integer size) {
        var valuesSourceBuilder =
                new TermsValuesSourceBuilder("appName").field("app.name.keyword");
        var aggregationBuilder = new CompositeAggregationBuilder("appName_bucket", List.of(valuesSourceBuilder));
        aggregationBuilder.size(ObjectUtil.defaultIfNull(size, 20));

        var nativeSearch = new NativeSearchQueryBuilder().withAggregations(aggregationBuilder)
                .build();
        nativeSearch.setMaxResults(0);
        SearchHits<LogStashDocument> searchHits = null;
        try {
            searchHits = restTemplate.search(nativeSearch, DOCUMENT_CLASS, indexCoordinates);
        } catch (Exception e) {
            throw new BusinessException("查询失败", e);
        }
        ElasticsearchAggregations aggregation = (ElasticsearchAggregations) searchHits.getAggregations();

        List<Aggregation> aggList = aggregation == null ? null : aggregation.aggregations().asList();
        if (CollectionUtils.isEmpty(aggList)) {
            return Collections.emptyList();
        }

        return aggList.stream()
                .flatMap(agg ->
                        ((CompositeAggregation) agg).getBuckets().stream()
                                .map(t -> t.getKey().get("appName").toString())
                ).collect(Collectors.toList());
    }

    private SearchHits<LogStashDocument> searchHits(LogStashQueryParam queryParam) {
        var queryBuilder = buildQueryBuilder(queryParam);
        withPageable(queryBuilder, queryParam);
        withSort(queryBuilder);

        IndexCoordinates indexCoordinates = null;
        try {
            indexCoordinates = obtainIndexCoordinates(queryParam.getStartTime(), queryParam.getEndTime());
        } catch (Exception e) {
            throw new BusinessException("查询日志失败", e);
        }
        if (indexCoordinates == null) {
            return null;
        }

        try {
            return restTemplate.search(queryBuilder.build(), DOCUMENT_CLASS, indexCoordinates);
        } catch (Exception e) {
            throw new BusinessException("查询日志失败", e);
        }
    }

    private IndexCoordinates obtainIndexCoordinates(LocalDateTime startTime, LocalDateTime endTime) {
        if (startTime == null && endTime == null) {
            return IndexCoordinates.of(convertIndex(LocalDateTime.now()));
        }

        LocalDateTime now = LocalDateTime.now();
        LocalDateTime end = null;
        if (endTime == null || endTime.isAfter(now)) {
            end = now.with(LocalTime.MAX);
        } else {
            end = endTime.with(LocalTime.MAX);
        }
        String[] indexNames = Stream.iterate(startTime == null ? LocalDateTime.now() : startTime, end::isAfter, f -> f.plusDays(1))
                .map(this::convertIndex)
                .filter(this::existsIndex)
                .toArray(String[]::new);
        if (indexNames.length == 0) {
            // 没有符合条件的索引
            logger.info("没有符合条件的索引存在");
            return null;
        }

        logger.info("日志查询索引：{}", Arrays.toString(indexNames));
        return IndexCoordinates.of(indexNames);
    }

    private String convertIndex(LocalDateTime date) {
        return indexPrefix + date.format(FORMATTER_INDEX);
    }

    private void withSort(NativeSearchQueryBuilder queryBuilder) {
        queryBuilder.withSorts(SortBuilders.fieldSort("@timestamp").order(SortOrder.DESC));
    }

    private void withPageable(NativeSearchQueryBuilder queryBuilder, LogStashQueryParam queryParam) {
        int page = Math.max(ObjectUtil.defaultIfNull(queryParam.getPage(), 1) - 1, 0);
        int pageSize = Math.max(ObjectUtil.defaultIfNull(queryParam.getPageSize(), 20), 1);

        queryBuilder.withPageable(PageRequest.of(page, pageSize));
    }

    private NativeSearchQueryBuilder buildQueryBuilder(LogStashQueryParam queryParam) {
        var filter = QueryBuilders.boolQuery();
        if (StrUtil.isNotBlank(queryParam.getAppName())) {
            filter.must(QueryBuilders.termsQuery("app.name.keyword", queryParam.getAppName()));
        }
        if (queryParam.getLogLevel() != null) {
            filter.must(QueryBuilders.termsQuery("level.keyword", queryParam.getLogLevel()));
        } else if (queryParam.getLogLevelMin() != null) {
            if (queryParam.getLogLevelMin() != LogLevel.TRACE) {
                filter.must(QueryBuilders.termsQuery("level.keyword", getLogLevels(queryParam.getLogLevelMin())));
            }
        }
        if (StrUtil.isNotBlank(queryParam.getTraceId())) {
            filter.must(QueryBuilders.termsQuery("traceId.keyword", queryParam.getTraceId()));
        }
        if (StrUtil.isNotBlank(queryParam.getThreadId())) {
            filter.must(QueryBuilders.termsQuery("thread.keyword", queryParam.getThreadId()));
        }
        if (queryParam.getStartTime() != null || queryParam.getEndTime() != null) {
            var rangeQuery = QueryBuilders.rangeQuery("@timestamp");
            if (queryParam.getStartTime() != null) {
                rangeQuery.from(queryParam.getStartTime());
            }
            if (queryParam.getEndTime() != null) {
                rangeQuery.to(queryParam.getEndTime());
            }
            filter.must(rangeQuery);
        }

        var queryBuilder = new NativeSearchQueryBuilder();
        queryBuilder.withFilter(filter);

        // 关键词搜索
        if (StrUtil.isNotBlank(queryParam.getKeyword())) {
            queryBuilder.withQuery(QueryBuilders.boolQuery().must(QueryBuilders.matchPhraseQuery("msg", queryParam.getKeyword())));

            // 高亮标签
            var highlight = queryParam.getHighlight();
            if (highlight != null) {
                if (StrUtil.isAllNotBlank(highlight.getPreTag(), highlight.getPostTag())) {
                    HighlightBuilder highlightBuilder = new HighlightBuilder()
//                            .field("msg")
                            .field("message")
                            .preTags(highlight.getPreTag())
                            .postTags(highlight.getPostTag());

                    queryBuilder.withHighlightBuilder(highlightBuilder);
                }
            }
        }

        return queryBuilder;
    }

    private boolean existsIndex(String indexName) {
        return restTemplate.indexOps(IndexCoordinates.of(indexName)).exists();
    }

    private List<LogLevel> getLogLevels(LogLevel min) {
        switch (min) {
            case TRACE:
                return List.of(LogLevel.TRACE, LogLevel.DEBUG, LogLevel.INFO, LogLevel.WARN, LogLevel.ERROR, LogLevel.FATAL, LogLevel.OFF);
            case DEBUG:
                return List.of(LogLevel.DEBUG, LogLevel.INFO, LogLevel.WARN, LogLevel.ERROR, LogLevel.FATAL, LogLevel.OFF);
            case INFO:
                return List.of(LogLevel.INFO, LogLevel.WARN, LogLevel.ERROR, LogLevel.FATAL, LogLevel.OFF);
            case WARN:
                return List.of(LogLevel.WARN, LogLevel.ERROR, LogLevel.FATAL, LogLevel.OFF);
            case ERROR:
                return List.of(LogLevel.ERROR, LogLevel.FATAL, LogLevel.OFF);
            case FATAL:
                return List.of(LogLevel.FATAL, LogLevel.OFF);
            case OFF:
                return List.of(LogLevel.OFF);
            default:
                throw new BusinessException("未知的日志级别");
        }
    }

    private void init() {
        if (client == null) {
            logger.error("日志查询组件初始化失败，未获取到有效的Elasticsearch Client");
            return;
        }
        this.indexPrefix = "filebeat-" + (CharSequenceUtil.isBlank(this.preIndex) ? "" : this.preIndex + "-");
        this.restTemplate = new ElasticsearchRestTemplate(client);
        this.restTemplate.setEntityCallbacks(new EntityCallbacks() {
            @Override
            public void addEntityCallback(@NonNull EntityCallback<?> callback) {

            }

            @NonNull
            @Override
            public <T> T callback(@NonNull Class<? extends EntityCallback> callbackType, @NonNull T entity, @NonNull Object... args) {
                Document document = (Document) args[0];
                ((LogStashDocument) entity).setAppName(document.getString("app.name"));
                return entity;
            }
        });
    }

    private ElasticsearchRestTemplate restTemplate;
    private String indexPrefix;
    private static final Class<LogStashDocument> DOCUMENT_CLASS = LogStashDocument.class;
    private static final DateTimeFormatter FORMATTER_INDEX = DateTimeFormatter.ofPattern("yyyy.MM.dd");
    private static final DateTimeFormatter FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss.SSS");
}
