package com.elitescloud.cloudt.system.cas;

import com.elitescloud.boot.auth.cas.model.thirdparty.WechatMiniProgramDTO;
import com.elitescloud.boot.auth.cas.model.thirdparty.WechatServiceDTO;
import com.elitescloud.boot.auth.cas.model.thirdparty.WechatSubscriptionDTO;
import com.elitescloud.boot.auth.cas.provider.ThirdPartyAccountTransferHelper;
import com.elitescloud.boot.auth.model.Result;
import com.elitescloud.cloudt.system.service.ThirdPartAccountMngService;
import com.elitescloud.cloudt.system.service.callback.ThirdPartyAccountChangedCallback;
import com.elitescloud.cloudt.system.service.common.constant.ThirdPartAccountType;
import com.elitescloud.cloudt.system.service.common.constant.ThirdPartyConstant;
import com.elitescloud.cloudt.system.service.model.entity.SysThirdPartyAccountDO;
import com.elitescloud.cloudt.system.service.repo.ThirdPartyAccountRepoProc;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.TaskExecutor;

import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

/**
 * 外部账号变更后的回调.
 *
 * @author Kaiser（wang shao）
 * @date 2/14/2023
 */
@Log4j2
class ThirdPartyAccountChangedCallbackCas implements ThirdPartyAccountChangedCallback {

    @Autowired
    private ThirdPartAccountMngService mngService;
    @Autowired
    private ThirdPartyAccountRepoProc repoProc;
    @Autowired
    private TaskExecutor taskExecutor;
    @Autowired
    private ObjectMapper objectMapper;

    private final ThirdPartyAccountTransferHelper helper;

    public ThirdPartyAccountChangedCallbackCas(ThirdPartyAccountTransferHelper thirdPartyAccountTransferHelper) {
        this.helper = thirdPartyAccountTransferHelper;
    }

    @Override
    public void onUpsert(boolean add, SysThirdPartyAccountDO accountDO) {
        this.execute(() -> {
            if (accountDO == null) {
                return null;
            }
            this.upsertWechat(accountDO);
            return accountDO.getId();
        });
    }

    @Override
    public void onEnabled(Long accountId, boolean enabled) {
        this.execute(() -> {
            var casId = repoProc.getCasId(accountId);
            if (casId != null) {
                helper.updateEnabled(casId, enabled);
                return accountId;
            }

            // 保存
            this.onUpsert(true, repoProc.get(accountId));
            return accountId;
        });
    }

    @Override
    public void onDelete(SysThirdPartyAccountDO accountDO) {
        this.execute(() -> {
            if (accountDO == null) {
                return null;
            }

            var casId = accountDO.getCasId();
            if (casId != null) {
                helper.delete(casId);
                return accountDO.getId();
            }

            return accountDO.getId();
        });
    }

    private void upsertWechat(SysThirdPartyAccountDO accountDO) {
        String type = accountDO.getAccountType();
        Map<String, String> config = null;
        try {
            config = objectMapper.readValue(accountDO.getConfigJson(), new TypeReference<>() {
            });
        } catch (JsonProcessingException e) {
            throw new IllegalStateException("读取账号配置异常", e);
        }

        Result<Long> result = null;
        if (type.equals(ThirdPartAccountType.WECHAT_SERVICE.getValue())) {
            WechatServiceDTO dto = new WechatServiceDTO();
            dto.setId(accountDO.getCasId());
            dto.setAppId(accountDO.getAccount());
            dto.setAppSecret(config.get(ThirdPartyConstant.CFG_WECHAT_APP_SECRET));
            dto.setName(accountDO.getName());
            dto.setEnabled(accountDO.getEnabled());

            result = helper.upsertWechatService(dto);
        } else if (type.equals(ThirdPartAccountType.WECHAT_MINI_APP.getValue())) {
            WechatMiniProgramDTO dto = new WechatMiniProgramDTO();
            dto.setId(accountDO.getCasId());
            dto.setAppId(accountDO.getAccount());
            dto.setAppSecret(config.get(ThirdPartyConstant.CFG_WECHAT_APP_SECRET));
            dto.setName(accountDO.getName());
            dto.setEnabled(accountDO.getEnabled());

            result = helper.upsertWechatMiniProgram(dto);
        } else if (type.equals(ThirdPartAccountType.WECHAT_SUBSCRIPTION.getValue())) {
            WechatSubscriptionDTO dto = new WechatSubscriptionDTO();
            dto.setId(accountDO.getCasId());
            dto.setAppId(accountDO.getAccount());
            dto.setAppSecret(config.get(ThirdPartyConstant.CFG_WECHAT_APP_SECRET));
            dto.setName(accountDO.getName());
            dto.setEnabled(accountDO.getEnabled());

            result = helper.upsertWechatSubscription(dto);
        } else {
            log.error("暂不支持的账号类型：{}", type);
            return;
        }

        if (Boolean.FALSE.equals(result.getSuccess())) {
            log.error("保存第三方账号到统一认证服务失败：{}", result.getMsg());
            return;
        }

        // 回调更新本地的ID
        if (result.getData() != null && accountDO.getCasId() == null) {
            mngService.updateCasId(accountDO.getId(), result.getData());
        }
    }

    private void execute(Supplier<Long> supplier) {
        CompletableFuture.runAsync(supplier::get, taskExecutor)
                .whenComplete((r, e) -> {
                    if (e != null) {
                        log.error("同步第三方账号异常：", e);
                    }
                });
    }
}
