package com.elitescloud.cloudt.system.cas;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.auth.cas.common.PwdRecordTypeEnum;
import com.elitescloud.boot.auth.cas.model.AuthPwdUpdateDTO;
import com.elitescloud.boot.auth.cas.provider.UserTransferHelper;
import com.elitescloud.boot.auth.config.AuthorizationSdkProperties;
import com.elitescloud.boot.auth.model.UpdaterInfoDTO;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.service.IUserService;
import com.elitescloud.cloudt.system.service.callback.UserChangedCallback;
import com.elitescloud.cloudt.system.service.model.bo.SysUserSaveBO;
import com.elitescloud.cloudt.system.service.model.entity.SysUserDO;
import com.elitescloud.cloudt.system.service.repo.UserRepoProc;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;

import javax.servlet.http.HttpServletRequest;
import java.util.Map;
import java.util.function.Supplier;

/**
 * 用户更新后的回调.
 *
 * @author Kaiser（wang shao）
 * 2022/12/3
 */
@Log4j2
class UserChangedCallbackCas extends BaseCasClientService implements UserChangedCallback {
    private final UserTransferHelper userTransferHelper;
    private final AuthorizationSdkProperties sdkProperties;

    @Autowired
    private UserRepoProc userRepoProc;
    @Autowired
    private IUserService userService;

    public UserChangedCallbackCas(AuthorizationSdkProperties sdkProperties, UserTransferHelper userTransferHelper) {
        this.userTransferHelper = userTransferHelper;
        this.sdkProperties = sdkProperties;
    }

    @Override
    public void onUpsert(boolean add, SysUserSaveBO saveBO, SysUserDO userDO) {
        var request = HttpServletUtil.currentRequest();
        execute(() -> {
            if (!ObjectUtil.defaultIfNull(sdkProperties.getCasClient().getEnabled(), false)) {
                // 禁用
                return null;
            }

            var dto = super.do2Dto(userDO, saveBO.getPassword());
            UpdaterInfoDTO updaterInfoDTO = this.createUpdaterInfo(request, null);
            dto.setUpdaterInfo(updaterInfoDTO);
            var syncResult = userTransferHelper.upsertUser(dto);

            if (!syncResult.getSuccess() || syncResult.getData() == null) {
                log.error("CAS同步用户账号信息失败：{}", syncResult);
                throw new BusinessException(CharSequenceUtil.blankToDefault(syncResult.getMsg(), "创建账号失败"));
            }

            // 更新认证中心用户ID
            if (ObjectUtil.equals(userDO.getCasUserId(), syncResult.getData())) {
                return null;
            }
            userService.updateCasSyncResult(Map.of(userDO.getUsername(), syncResult.getData()));

            return null;
        });
    }

    @Override
    public void onEnabled(Long userId, boolean enabled) {
        execute(() -> {
            if (!ObjectUtil.defaultIfNull(sdkProperties.getCasClient().getEnabled(), false)) {
                // 禁用
                return null;
            }

            var id = userRepoProc.getCasUserId(userId);
            if (id == null) {
                // 暂不同步，由定时任务统一执行
                return null;
            }

            // 更新状态
            var userEnabled = userRepoProc.getEnabled(userId);
            userTransferHelper.updateEnabled(id, ObjectUtil.defaultIfNull(userEnabled, false));
            return null;
        });
    }

    @Override
    public void onUpdatePassword(Long userId, String password, String originalPassword) {
        var request = HttpServletUtil.currentRequest();
        execute(() -> {
            if (!ObjectUtil.defaultIfNull(sdkProperties.getCasClient().getEnabled(), false)) {
                // 禁用
                return null;
            }

            var id = userRepoProc.getCasUserId(userId);
            if (id == null) {
                // 暂不同步，由定时任务统一执行
                return null;
            }

            // 更新密码
            AuthPwdUpdateDTO updateDTO = new AuthPwdUpdateDTO();
            updateDTO.setUserId(id);
            updateDTO.setPassword(password);

            var currentUser = SecurityContextUtil.currentUser();
            if (currentUser == null) {
                updateDTO.setUpdateType(PwdRecordTypeEnum.RETRIEVE);
            } else if (currentUser.getUserId().longValue() == userId) {
                updateDTO.setUpdateType(PwdRecordTypeEnum.UPDATE);
            } else {
                updateDTO.setUpdateType(PwdRecordTypeEnum.RESET);
            }

            // 操作人信息
            UpdaterInfoDTO updaterInfoDTO = this.createUpdaterInfo(request, currentUser);
            updateDTO.setUpdaterInfo(updaterInfoDTO);
            userTransferHelper.updatePwd(updateDTO);
            return null;
        });
    }

    @Override
    public void onUpdateMobile(Long userId, String mobile) {
        execute(() -> {
            if (!ObjectUtil.defaultIfNull(sdkProperties.getCasClient().getEnabled(), false)) {
                // 禁用
                return null;
            }

            var id = userRepoProc.getCasUserId(userId);
            if (id == null) {
                // 暂不同步，由定时任务统一执行
                return null;
            }

            // 更新手机号
            userTransferHelper.updateMobile(id, mobile);
            return null;
        });
    }

    @Override
    public void onUpdateEmail(Long userId, String email) {
        execute(() -> {
            if (!ObjectUtil.defaultIfNull(sdkProperties.getCasClient().getEnabled(), false)) {
                // 禁用
                return null;
            }

            var id = userRepoProc.getCasUserId(userId);
            if (id == null) {
                // 暂不同步，由定时任务统一执行
                return null;
            }

            // 更新邮箱
            userTransferHelper.updateEmail(id, email);
            return null;
        });
    }

    @Override
    public void onDelete(Long userId) {
        execute(() -> {
            if (!ObjectUtil.defaultIfNull(sdkProperties.getCasClient().getEnabled(), false)) {
                // 禁用
                return null;
            }

            var id = userRepoProc.getCasUserId(userId);
            if (id == null) {
                // 暂不同步，由定时任务统一执行
                return null;
            }

            // 删除
            userTransferHelper.delete(id);
            return null;
        });
    }

    private UpdaterInfoDTO createUpdaterInfo(HttpServletRequest request, GeneralUserDetails currentUser) {
        if (request == null) {
            request = HttpServletUtil.currentRequest();
        }
        if (currentUser == null) {
            currentUser = SecurityContextUtil.currentUser();
        }
        UpdaterInfoDTO updaterInfoDTO = new UpdaterInfoDTO();

        if (currentUser != null) {
            updaterInfoDTO.setUpdaterId(currentUser.getUser().getCasUserId());
            updaterInfoDTO.setUpdater(currentUser.getUser().getUsername());
        }

        if (request != null) {
            updaterInfoDTO.setIp(HttpServletUtil.currentClientIp());
            updaterInfoDTO.setBrowserAgent(request.getHeader(HttpHeaders.USER_AGENT));
        }
        return updaterInfoDTO;
    }

    private <T> void execute(Supplier<T> supplier) {
        supplier.get();
    }
}
