package com.elitescloud.cloudt.system.controller.mng.common;

import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.model.vo.CommonTenantAppVO;
import com.elitescloud.cloudt.system.model.vo.query.common.CommonAppPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.common.CommonAreaQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.extend.CurrencyRateQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.common.CommonAreaTreeRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.common.CommonTaxRateRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.system.PwdStrategyRespVO;
import com.elitescloud.cloudt.system.service.SystemQueryService;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;

/**
 * 通用系统接口.
 *
 * @author Kaiser（wang shao）
 * 2022/10/11
 */
@Api(tags = "通用系统接口")
@RestController
@RequestMapping(value = "/mng/common/system", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
public class CommonSystemController {

    private final SystemQueryService systemQueryService;

    public CommonSystemController(SystemQueryService systemQueryService) {
        this.systemQueryService = systemQueryService;
    }

    /**
     * 应用列表
     *
     * @return 应用列表
     */
    @ApiOperation(value = "应用列表")
    @ApiOperationSupport(order = 1)
    @GetMapping(value = "/apps")
    public ApiResult<List<CommonTenantAppVO>> listApp() {
        return systemQueryService.listApp();
    }

    /**
     * 应用分页列表
     *
     * @param queryVO 查询参数
     * @return 应用列表
     */
    @ApiOperation(value = "应用分页列表")
    @ApiOperationSupport(order = 1)
    @PostMapping(value = "/apps/page")
    public ApiResult<PagingVO<CommonTenantAppVO>> pageApp(@RequestBody CommonAppPageQueryVO queryVO) {
        return systemQueryService.pageApp(queryVO);
    }

    /**
     * 货币列表
     *
     * @return 货币列表
     */
    @ApiOperation(value = "货币列表")
    @ApiOperationSupport(order = 2)
    @GetMapping(value = "/currency")
    public ApiResult<List<CodeNameParam>> listCurrency() {
        return systemQueryService.listCurrency();
    }

    /**
     * 税率列表
     *
     * @return 税率列表
     */
    @ApiOperation(value = "税率列表")
    @ApiOperationSupport(order = 2)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "taxRateNo", value = "税率编码"),
            @ApiImplicitParam(name = "taxRateType", value = "税率类型, UDC[cloudt-system:taxRateType]"),
    })
    @GetMapping(value = "/taxRate")
    public ApiResult<List<CommonTaxRateRespVO>> listTaxRate(@RequestParam(name = "taxRateNo", required = false) String taxRateNo,
                                                            @RequestParam(name = "taxRateType", required = false) String taxRateType) {
        return systemQueryService.listTaxRate(taxRateNo, taxRateType);
    }

    /**
     * 查询汇率
     *
     * @return 税率列表
     */
    @ApiOperation(value = "查询汇率")
    @ApiOperationSupport(order = 3)
    @PostMapping(value = "/currencyRatio")
    public ApiResult<Double> findTaxRate(@RequestBody @Valid CurrencyRateQueryVO queryVO) {
        return systemQueryService.queryCurrentRate(queryVO);
    }

    /**
     * 行政区域列表
     *
     * @return 行政区域列表
     */
    @ApiOperation(value = "行政区域列表")
    @ApiOperationSupport(order = 5)
    @GetMapping(value = "/area")
    public ApiResult<List<CommonAreaTreeRespVO>> listArea(CommonAreaQueryVO queryVO) {
        return systemQueryService.listArea(queryVO);
    }

    @ApiOperation(value = "获取密码策略")
    @ApiOperationSupport(order = 4)
    @GetMapping(value = "/pwdStrategy")
    public ApiResult<List<PwdStrategyRespVO>> listStrategy() {
        return systemQueryService.listPwdStrategyForCreateAccount();
    }
}
