package com.elitescloud.cloudt.system.controller.mng.extend;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.model.vo.resp.extend.AlertRespVO;
import com.elitescloud.cloudt.system.model.vo.save.extend.AlertSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.extend.AlertSendSaveVO;
import com.elitescloud.cloudt.system.service.SysAlertMngService;
import com.elitescloud.cloudt.system.service.SysAlertService;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;

/**
 * 预警配置管理.
 *
 * @author Kaiser（wang shao）
 * @date 2023/10/26
 */
@Api(tags = "预警配置管理")
@RestController
@RequestMapping(value = "/mng/alert", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
public class AlertMngController {

    private SysAlertMngService service;
    private SysAlertService alertService;

    /**
     * 根据业务类型获取模板参数
     *
     * @param businessType 业务类型
     * @return 模板参数
     */
    @GetMapping(value = "/getTmplParam")
    @ApiOperation(value = "根据业务类型获取模板参数")
    @ApiOperationSupport(order = 2)
    @ApiImplicitParam(name = "businessType", value = "业务类型", required = true)
    public ApiResult<List<CodeNameParam>> getTmplParam(@RequestParam("businessType") String businessType) {
        return service.getTmplParam(businessType);
    }

    /**
     * 保存预警配置
     *
     * @param saveVO 配置参数
     * @return 配置参数ID
     */
    @PostMapping(value = "/save")
    @ApiOperation(value = "保存预警配置")
    @ApiOperationSupport(order = 1)
    public ApiResult<Long> save(@RequestBody @Valid AlertSaveVO saveVO) {
        return service.saveAlert(saveVO);
    }

    /**
     * 根据业务类型获取预警配置
     *
     * @param businessType 业务类型
     * @return 预警配置
     */
    @GetMapping(value = "/get")
    @ApiOperation(value = "根据业务类型获取预警配置")
    @ApiOperationSupport(order = 2)
    @ApiImplicitParam(name = "businessType", value = "业务类型", required = true)
    public ApiResult<AlertRespVO> get(@RequestParam("businessType") String businessType) {
        return service.getAlert(businessType);
    }

    /**
     * 测试预警发送
     *
     * @param sendSaveVO 发送参数
     * @return 发送结果
     */
    @PostMapping(value = "/test/send")
    @ApiOperation(value = "测试预警发送")
    @ApiOperationSupport(order = 3)
    public ApiResult<Boolean> testSend(@RequestBody @Valid AlertSendSaveVO sendSaveVO) {
        if (CharSequenceUtil.isNotBlank(sendSaveVO.getContent())) {
            return alertService.sendAlert(sendSaveVO.getBusinessType(), sendSaveVO.getCategory(), sendSaveVO.getContent());
        }
        if (CollUtil.isNotEmpty(sendSaveVO.getTmplParams())) {
            return alertService.sendAlertByTmpl(sendSaveVO.getBusinessType(), sendSaveVO.getCategory(), sendSaveVO.getTmplParams());
        }

        return ApiResult.fail("发送内容和模板参数至少一个不为空");
    }

    @Autowired
    public void setService(SysAlertMngService service) {
        this.service = service;
    }

    @Autowired
    public void setAlertService(SysAlertService alertService) {
        this.alertService = alertService;
    }
}
