package com.elitescloud.cloudt.system.controller.mng.extend;

import com.elitescloud.boot.log.common.BusinessObject;
import com.elitescloud.boot.log.common.BusinessObjectOperation;
import com.elitescloud.boot.log.common.OperationTypeEnum;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.platform.model.vo.extend.resp.AreaDetailRespVO;
import com.elitescloud.cloudt.platform.model.vo.extend.resp.AreaMngTreeRespVO;
import com.elitescloud.cloudt.platform.model.vo.extend.save.PlatformAreaSaveVO;
import com.elitescloud.cloudt.system.service.AreaMngService;
import com.elitescloud.cloudt.system.service.common.constant.BusinessObjectConstant;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;

/**
 * 行政区域管理.
 *
 * @author Kaiser（wang shao）
 * 2023/1/10
 */
@Api(tags = "行政区域管理")
@RestController
@RequestMapping(value = "/mng/area", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
@BusinessObject(businessType = BusinessObjectConstant.SYS_AREA)
public class AreaController {

    private final AreaMngService service;

    public AreaController(AreaMngService service) {
        this.service = service;
    }

    /**
     * 获取是否启用平台设置
     *
     * @return 是否启用平台设置
     */
    @GetMapping(value = "/platform/enabled")
    @ApiOperation(value = "获取是否启用平台设置")
    @ApiOperationSupport(order = 1)
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "获取是否启用平台设置"
    )
    public ApiResult<Boolean> getEnabledPlatform() {
        return service.getEnabledPlatform();
    }

    /**
     * 切换是否启用平台设置
     *
     * @return 是否启用平台设置
     */
    @PutMapping(value = "/platform/enabled")
    @ApiOperation(value = "切换是否启用平台设置")
    @ApiOperationSupport(order = 2)
    @BusinessObjectOperation(operationType = OperationTypeEnum.UPDATE, operationDescription = "切换是否启用平台设置"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Boolean> updateEnabledPlatform() {
        return service.updateEnabledPlatform();
    }

    /**
     * 行政区域树
     *
     * @param tree 是否返回树状，默认false
     * @return 行政区域树列表
     */
    @GetMapping(value = "/tree")
    @ApiOperation(value = "行政区域树")
    @ApiOperationSupport(order = 3)
    @ApiImplicitParam(name = "tree", value = "是否返回树状，默认false", defaultValue = "false")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "行政区域树")
    public ApiResult<List<AreaMngTreeRespVO>> tree(@RequestParam(name = "tree", defaultValue = "false") Boolean tree) {
        return service.tree(tree);
    }

    /**
     * 保存行政区域
     *
     * @param saveVO 行政区域参数
     * @return 行政区域ID
     */
    @PostMapping(value = "/save")
    @ApiOperation(value = "保存行政区域")
    @ApiOperationSupport(order = 4)
    @BusinessObjectOperation(operationType = OperationTypeEnum.UPDATE, operationDescription = "切换是否启用平台设置"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> save(@RequestBody @Valid PlatformAreaSaveVO saveVO) {
        return service.save(saveVO);
    }

    /**
     * 获取行政区域详情
     *
     * @param id 行政区域ID
     * @return 行政区域详情
     */
    @GetMapping(value = "/{id}")
    @ApiOperation(value = "获取行政区域详情")
    @ApiOperationSupport(order = 5)
    @ApiImplicitParam(name = "id", value = "区域ID", required = true)
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "获取行政区域详情"
    )
    public ApiResult<AreaDetailRespVO> get(@PathVariable("id") Long id) {
        return service.get(id);
    }

    /**
     * 删除行政区域详情
     *
     * @param id 行政区域ID
     * @return 行政区域ID
     */
    @DeleteMapping(value = "/{id}")
    @ApiOperation(value = "删除行政区域详情")
    @ApiOperationSupport(order = 6)
    @ApiImplicitParam(name = "id", value = "区域ID", required = true)
    @BusinessObjectOperation(operationType = OperationTypeEnum.DELETE, operationDescription = "删除行政区域详情"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> delete(@PathVariable("id") Long id) {
        return service.delete(id);
    }

    /**
     * 重构行政区域树
     *
     * @param rootId 根区域ID
     * @return 行政区域ID
     */
    @PostMapping(value = "/tree/rebuild/{rootId}")
    @ApiOperation(value = "重构行政区域树")
    @ApiOperationSupport(order = 7)
    @ApiImplicitParam(name = "rootId", value = "根区域ID")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "重构行政区域树")
    public ApiResult<Boolean> rebuildTree(@PathVariable("rootId") Long rootId) {
        return service.rebuildTree(rootId);
    }
}
