package com.elitescloud.cloudt.system.controller.mng.extend;

import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.log.common.BusinessObject;
import com.elitescloud.boot.log.common.BusinessObjectOperation;
import com.elitescloud.boot.log.common.OperationTypeEnum;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.constant.CurrencyRateCalcMethod;
import com.elitescloud.cloudt.system.model.vo.query.extend.CurrencyRatePageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.extend.CurrencyRateDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.extend.CurrencyRatePageRespVO;
import com.elitescloud.cloudt.system.model.vo.save.extend.CurrencyRateSaveVO;
import com.elitescloud.cloudt.system.service.CurrencyRateMngService;
import com.elitescloud.cloudt.system.service.common.constant.BusinessObjectConstant;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 汇率管理.
 *
 * @author Kaiser（wang shao）
 * 2023/1/13
 */
@Api(tags = "汇率管理")
@RestController
@RequestMapping(value = "/mng/currencyRate", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
@BusinessObject(businessType = BusinessObjectConstant.SYS_CURRENCY_RATE)
public class CurrencyRateController {

    private final CurrencyRateMngService service;

    public CurrencyRateController(CurrencyRateMngService service) {
        this.service = service;
    }

    /**
     * 分页查询汇率
     *
     * @param queryVO 查询参数
     * @return 汇率列表
     */
    @PostMapping(value = "/q")
    @ApiOperation(value = "分页查询汇率")
    @ApiOperationSupport(order = 1)
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT_PAGE, operationDescription = "分页查询汇率"
    )
    public ApiResult<PagingVO<CurrencyRatePageRespVO>> page(@RequestBody CurrencyRatePageQueryVO queryVO) {
        return service.page(queryVO);
    }

    /**
     * 切换启用禁用
     *
     * @param id 汇率ID
     * @return 切换结果
     */
    @PatchMapping(value = "/switch/{id}/enabled")
    @ApiOperation(value = "切换启用禁用")
    @ApiOperationSupport(order = 2)
    @ApiImplicitParam(name = "id", value = "汇率ID", required = true)
    @BusinessObjectOperation(operationType = OperationTypeEnum.UPDATE, operationDescription = "切换启用禁用"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> switchEnabled(@PathVariable("id") Long id) {
        return service.updateEnabled(id);
    }

    /**
     * 获取汇率详情
     *
     * @param id 汇率ID
     * @return 汇率详情
     */
    @GetMapping(value = "/{id}")
    @ApiOperation(value = "获取汇率详情")
    @ApiOperationSupport(order = 3)
    @ApiImplicitParam(name = "id", value = "汇率ID", required = true)
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "获取汇率详情"
    )
    public ApiResult<CurrencyRateDetailRespVO> get(@PathVariable("id") Long id) {
        return service.get(id);
    }

    /**
     * 保存汇率
     *
     * @param saveVO 汇率信息
     * @return 汇率ID
     */
    @PostMapping(value = "/save")
    @ApiOperation(value = "保存汇率")
    @ApiOperationSupport(order = 4)
    @BusinessObjectOperation(operationType = OperationTypeEnum.SAVE, operationDescription = "保存汇率"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> save(@RequestBody @Valid CurrencyRateSaveVO saveVO) {
        return service.save(saveVO);
    }

    /**
     * 删除汇率信息
     *
     * @param id 汇率ID
     * @return 汇率ID
     */
    @DeleteMapping(value = "/{id}")
    @ApiOperation(value = "删除汇率信息")
    @ApiOperationSupport(order = 5)
    @ApiImplicitParam(name = "id", value = "汇率ID", required = true)
    @BusinessObjectOperation(operationType = OperationTypeEnum.DELETE, operationDescription = "删除汇率信息"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> delete(@PathVariable("id") Long id) {
        return service.delete(id);
    }

    /**
     * 算法类型
     *
     * @return 算法类型列表
     */
    @ApiOperation(value = "算法类型")
    @ApiOperationSupport(order = 11)
    @GetMapping(value = "/calcMethod")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "算法类型"
    )
    public ApiResult<List<CodeNameParam>> calcMethod() {
        var values = Arrays.stream(CurrencyRateCalcMethod.values())
                .map(t -> new CodeNameParam(t.name(), t.getDescription()))
                .collect(Collectors.toList());
        return ApiResult.ok(values);
    }
}
