package com.elitescloud.cloudt.system.controller.mng.extend;


import com.el.coordinator.boot.fsm.model.vo.FileObjRespVO;
import com.elitescloud.boot.common.param.FileInfoVO;
import com.elitescloud.boot.log.common.BusinessObject;
import com.elitescloud.boot.log.common.BusinessObjectOperation;
import com.elitescloud.boot.log.common.OperationTypeEnum;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.model.vo.query.extend.TmplImportQueryParam;
import com.elitescloud.cloudt.system.model.vo.query.extend.TmplQueryParam;
import com.elitescloud.cloudt.system.model.vo.resp.extend.TmplEditRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.extend.TmplImportRecordRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.extend.TmplPageMngRespVO;
import com.elitescloud.cloudt.system.model.vo.save.extend.TmplSaveVO;
import com.elitescloud.cloudt.system.service.TmplImportRecordService;
import com.elitescloud.cloudt.system.service.TmplMngService;
import com.elitescloud.cloudt.system.service.common.constant.BusinessObjectConstant;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import com.github.xiaoymin.knife4j.annotations.ApiSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpEntity;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.mvc.method.annotation.StreamingResponseBody;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.List;

/**
 * 导入导出模板管理.
 *
 * @author Kaiser（wang shao）
 * @date 2021/5/28
 */
@RestController
@RequestMapping(value = "/api/tmpl", produces = MediaType.APPLICATION_JSON_VALUE)
@Api(tags = "导入导出模板管理")
@ApiSupport(author = "wang shao", order = 12)
@Validated
@Slf4j
@BusinessObject(businessType = BusinessObjectConstant.SYS_TMPL)
public class TmplController {

    @Autowired
    private TmplMngService tmplService;
    @Autowired
    private TmplImportRecordService tmplImportRecordService;

    @PostMapping("/search")
    @ApiOperation(value = "分页查询模板")
    @ApiOperationSupport(order = 1)
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT_PAGE, operationDescription = "分页查询模板"
    )
    public ApiResult<PagingVO<TmplPageMngRespVO>> search(@RequestBody @NotNull(message = "查询参数为空") TmplQueryParam queryParam) {
        return tmplService.search(queryParam);
    }

    @PostMapping("/file")
    @ApiOperation(value = "上传模板文件")
    @ApiOperationSupport(order = 2)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "file", value = "模板文件", required = true)
    })
    @BusinessObjectOperation(operationType = OperationTypeEnum.UPLOAD, operationDescription = "上传模板文件"
            , logEnabled = true)
    public ApiResult<FileInfoVO> uploadTmpl(@NotNull(message = "模板文件为空") @RequestParam(name = "file") MultipartFile file) {
        return tmplService.uploadTmpl(file);
    }

    @PostMapping(value = "")
    @ApiOperation(value = "新增模板")
    @ApiOperationSupport(order = 3)
    @BusinessObjectOperation(operationType = OperationTypeEnum.ADD, operationDescription = "新增模板"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> save(@RequestBody @Valid TmplSaveVO saveVO) {
        return tmplService.save(saveVO);
    }

    @GetMapping(value = "/{id}")
    @ApiOperation(value = "查看模板详细信息")
    @ApiOperationSupport(order = 4)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "模板ID", required = true, paramType = "path", dataType = "long")
    })
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "查看模板详细信息"
    )
    public ApiResult<TmplEditRespVO> get(@PathVariable Long id) {
        return tmplService.get(id);
    }

    @PutMapping(value = "/{id}")
    @ApiOperation(value = "更新模板")
    @ApiOperationSupport(order = 5)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "模板ID", required = true, paramType = "path", dataType = "long")
    })
    @BusinessObjectOperation(operationType = OperationTypeEnum.UPDATE, operationDescription = "更新模板"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> update(@PathVariable Long id, @RequestBody @Valid TmplSaveVO saveVO) {
        return tmplService.update(id, saveVO);
    }

    @DeleteMapping(value = "/{id}")
    @ApiOperation(value = "删除模板")
    @ApiOperationSupport(order = 6)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "模板ID", required = true, paramType = "path", dataType = "long")
    })
    @BusinessObjectOperation(operationType = OperationTypeEnum.DELETE, operationDescription = "删除模板"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> delete(@PathVariable Long id) {
        return tmplService.delete(id);
    }

    @PatchMapping("/{id}/enabled")
    @ApiOperation(value = "启用、停用模板")
    @ApiOperationSupport(order = 7)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "模板ID", required = true, paramType = "path", dataType = "long")
    })
    @BusinessObjectOperation(operationType = OperationTypeEnum.UPDATE, operationDescription = "启用/停用模板"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> enable(@PathVariable Long id) {
        return tmplService.updateEnabled(id);
    }

    @GetMapping(value = "/file")
    @ApiOperation(value = "下载模板文件", produces = MediaType.APPLICATION_OCTET_STREAM_VALUE)
    @ApiOperationSupport(order = 8)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "fileCode", value = "模板文件的唯一标识", required = true, paramType = "query")
    })
    @BusinessObjectOperation(operationType = OperationTypeEnum.DOWNLOAD, operationDescription = "下载模板文件"
    )
    public HttpEntity<StreamingResponseBody> download(String fileCode) {
        return tmplService.downloadByFileCode(fileCode);
    }

    @GetMapping(value = "/file/byTmplCode")
    @ApiOperation(value = "根据模板编码下载模板文件", produces = MediaType.APPLICATION_OCTET_STREAM_VALUE)
    @ApiOperationSupport(order = 8)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "tmplCode", value = "模板文件编码", required = true, paramType = "query")
    })
    @BusinessObjectOperation(operationType = OperationTypeEnum.DOWNLOAD, operationDescription = "根据模板编码下载模板文件"
    )
    public HttpEntity<StreamingResponseBody> downloadByTmplCode(String tmplCode) {
        return tmplService.downloadByCode(tmplCode);
    }

    @DeleteMapping(value = "/file")
    @ApiOperation(value = "删除模板文件")
    @ApiOperationSupport(order = 9)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "fileCode", value = "模板文件的唯一标识", required = true, paramType = "query")
    })
    @BusinessObjectOperation(operationType = OperationTypeEnum.DELETE, operationDescription = "删除模板文件"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<String> deleteFile(@NotBlank(message = "文件标识为空") String fileCode) {
        return tmplService.deleteTmplFile(fileCode);
    }

    @PostMapping("/{id}/import")
    @ApiOperation(value = "分页查询导入（导出）记录")
    @ApiOperationSupport(order = 21)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "模板ID", required = true, paramType = "path", dataType = "long")
    })
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT_PAGE, operationDescription = "分页查询记录"
    )
    public ApiResult<PagingVO<TmplImportRecordRespVO>> searchImport(@PathVariable Long id, @RequestBody @NotNull(message = "查询参数为空") TmplImportQueryParam queryParam) {
        return tmplImportRecordService.searchImport(id, queryParam);
    }

    @GetMapping("/record/{recordId}/files")
    @ApiOperation(value = "获取导出的文件列表")
    @ApiOperationSupport(order = 22)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "importId", value = "导出记录ID", required = true, paramType = "path", dataType = "long")
    })
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT_PAGE, operationDescription = "获取导出的文件列表"
    )
    public ApiResult<List<FileObjRespVO<String>>> listFileOfExport(@PathVariable("recordId") Long recordId) {
        return tmplImportRecordService.listFileOfExport(recordId);
    }
}
