package com.elitescloud.cloudt.system.controller.mng.menu;

import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.model.vo.query.menu.MenuFavorPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.menu.MenuFavorRespVO;
import com.elitescloud.cloudt.system.model.vo.save.menu.MenuFavorSaveVO;
import com.elitescloud.cloudt.system.service.MenuFavorMngService;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;

/**
 * 收藏菜单管理.
 *
 * @author Kaiser（wang shao）
 * 2022/9/19
 */
@Api(tags = "收藏菜单管理")
@RestController
@RequestMapping(value = "/mng/menuFavor", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
public class MenuFavorMngController {

    private final MenuFavorMngService menuFavorMngService;

    public MenuFavorMngController(MenuFavorMngService menuFavorMngService) {
        this.menuFavorMngService = menuFavorMngService;
    }

    /**
     * 分页查询收藏菜单
     *
     * @return 菜单树
     */
    @ApiOperation(value = "分页查询收藏菜单")
    @ApiOperationSupport(order = 1)
    @PostMapping(value = "/page")
    public ApiResult<PagingVO<MenuFavorRespVO>> page(@RequestBody MenuFavorPageQueryVO queryVO) {
        return menuFavorMngService.pageMng(queryVO);
    }

    /**
     * 保存收藏菜单
     *
     * @param saveVO 菜单信息
     * @return 新增结果
     */
    @ApiOperation(value = "保存收藏菜单")
    @ApiOperationSupport(order = 2)
    @PostMapping(value = "/save")
    public ApiResult<Long> save(@RequestBody @Valid MenuFavorSaveVO saveVO) {
        return menuFavorMngService.upsert(saveVO);
    }

    /**
     * 获取收藏菜单
     *
     * @param id 菜单ID
     * @return 菜单信息
     */
    @ApiOperation(value = "获取收藏菜单")
    @ApiOperationSupport(order = 4)
    @ApiImplicitParam(name = "id", value = "收藏记录ID", required = true, dataType = "int")
    @GetMapping(value = "/get/{id}")
    public ApiResult<MenuFavorRespVO> get(@PathVariable("id") Long id) {
        return menuFavorMngService.get(id);
    }

    /**
     * 删除收藏菜单
     *
     * @param id 菜单ID
     * @return 删除结果
     */
    @ApiOperation(value = "删除收藏菜单")
    @ApiOperationSupport(order = 5)
    @ApiImplicitParam(name = "id", value = "收藏记录ID", required = true, dataType = "int")
    @DeleteMapping(value = "/del/{id}")
    public ApiResult<Long> delete(@PathVariable("id") Long id) {
        return menuFavorMngService.delete(id);
    }

    /**
     * 切换启用禁用
     *
     * @param id 菜单ID
     * @return 切换结果
     */
    @ApiOperation(value = "切换启用禁用")
    @ApiOperationSupport(order = 5)
    @ApiImplicitParam(name = "id", value = "收藏记录ID", required = true, dataType = "int")
    @PatchMapping(value = "/switch/enabled/{id}")
    public ApiResult<Long> switchEnabled(@PathVariable("id") Long id) {
        return menuFavorMngService.switchEnabled(id);
    }
}
