package com.elitescloud.cloudt.system.controller.mng.menu;

import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.model.vo.resp.menu.MenuTreeCustomRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.menu.MenuTreeRespVO;
import com.elitescloud.cloudt.system.model.vo.save.menu.MenuTreeSaveVO;
import com.elitescloud.cloudt.system.service.MenuMngService;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;

/**
 * 菜单管理.
 *
 * @author Kaiser（wang shao）
 * 2022/9/19
 */
@Api(tags = "菜单管理")
@RestController
@RequestMapping(value = "/mng/menu", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
public class MenuMngController {

    private final MenuMngService menuMngService;

    public MenuMngController(MenuMngService menuMngService) {
        this.menuMngService = menuMngService;
    }

    /**
     * 获取默认菜单树
     *
     * @return 菜单树
     */
    @ApiOperation(value = "获取默认菜单树")
    @ApiOperationSupport(order = 1)
    @ApiImplicitParam(name = "tree", value = "是否返回树形结构数据", dataType = "boolean", defaultValue = "true")
    @GetMapping(value = "/tree/default")
    public ApiResult<List<MenuTreeRespVO>> getTreeDefault(@RequestParam(name = "tree", defaultValue = "true") Boolean tree) {
        return menuMngService.getTreeDefault(tree);
    }

    /**
     * 获取自定义菜单树
     *
     * @return 菜单树
     */
    @ApiOperation(value = "获取自定义菜单树")
    @ApiOperationSupport(order = 2)
    @ApiImplicitParam(name = "tree", value = "是否返回树形结构数据", dataType = "boolean", defaultValue = "true")
    @GetMapping(value = "/tree/custom")
    public ApiResult<List<MenuTreeCustomRespVO>> getTreeCustom(@RequestParam(name = "tree", defaultValue = "true") Boolean tree) {
        return menuMngService.getTreeCustom(tree);
    }

    /**
     * 获取是否启用自定义菜单树
     *
     * @return 是否启用
     */
    @ApiOperation(value = "获取是否启用自定义菜单树")
    @ApiOperationSupport(order = 3)
    @GetMapping(value = "/tree/custom/enabled")
    public ApiResult<Boolean> getEnabledCustom() {
        return menuMngService.getEnabled();
    }

    /**
     * 保存菜单树
     *
     * @param saveVOList 菜单树
     * @return 保存结果
     */
    @ApiOperation(value = "保存菜单树")
    @ApiOperationSupport(order = 4)
    @PostMapping(value = "/tree/save")
    public ApiResult<Boolean> saveTree(@RequestBody @Valid List<MenuTreeSaveVO> saveVOList) {
        return menuMngService.saveTree(saveVOList);
    }

    /**
     * 切换是否启用自定义菜单树
     *
     * @return 更新结果
     */
    @ApiOperation(value = "切换是否启用自定义菜单树")
    @ApiOperationSupport(order = 11)
    @PatchMapping(value = "/customEnabled")
    public ApiResult<Boolean> switchEnabled() {
        return menuMngService.updateEnabled();
    }
}
