package com.elitescloud.cloudt.system.controller.mng.org;

import com.elitescloud.boot.log.common.BusinessObject;
import com.elitescloud.boot.log.common.BusinessObjectOperation;
import com.elitescloud.boot.log.common.OperationTypeEnum;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.model.vo.query.org.OrgPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.OrgDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.OrgPagedRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.OrgTreeNodeRespVO;
import com.elitescloud.cloudt.system.model.vo.save.org.OrgSaveVO;
import com.elitescloud.cloudt.system.service.OrgMngService;
import com.elitescloud.cloudt.system.service.common.constant.BusinessObjectConstant;
import com.elitescloud.cloudt.system.service.common.constant.BusinessObjectEnum;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;

/**
 * 组织管理.
 *
 * @author Kaiser（wang shao）
 * 2022/9/20
 */
@Api(tags = "组织管理")
@RestController
@RequestMapping(value = "/mng/org", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
@BusinessObject(businessType = BusinessObjectConstant.SYS_ORG)
public class OrgMngController {

    private final OrgMngService orgMngService;

    public OrgMngController(OrgMngService orgMngService) {
        this.orgMngService = orgMngService;
    }

    /**
     * 获取组织树
     *
     * @return 菜单树
     */
    @ApiOperation(value = "获取行政组织树")
    @ApiOperationSupport(order = 1)
    @ApiImplicitParam(name = "tree", value = "是否返回树形结构数据", dataType = "boolean", defaultValue = "true")
    @GetMapping(value = "/tree/get")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "获取行政组织树"
            , logEnabled = false, resultJsonEnabled = false, argsJsonEnabled = false)
    public ApiResult<List<OrgTreeNodeRespVO>> getTree(@RequestParam(name = "tree", defaultValue = "true") Boolean tree) {
        return orgMngService.getOrgTree(tree);
    }

    /**
     * 获取组织详情
     *
     * @param id 组织ID
     * @return 组织信息
     */
    @ApiOperation(value = "获取组织详情")
    @ApiOperationSupport(order = 2)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "组织ID", required = true, dataType = "int"),
            @ApiImplicitParam(name = "withParentType", value = "指定的上级类型，[UDC]cloudt-system:orgType")
    })
    @GetMapping(value = "/get/{id}")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "获取组织详情"
            , logEnabled = false, resultJsonEnabled = false, argsJsonEnabled = false)
    public ApiResult<OrgDetailRespVO> get(@PathVariable("id") Long id,
                                          @RequestParam(name = "withParentType", required = false) String withParentType) {
        return orgMngService.getDetail(id, withParentType);
    }

    /**
     * 保存组织
     *
     * @param saveVO 组织信息
     * @return 新增结果
     */
    @ApiOperation(value = "保存组织")
    @ApiOperationSupport(order = 3)
    @PostMapping(value = "/save")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SAVE, operationDescription = "保存组织"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> save(@RequestBody @Valid OrgSaveVO saveVO) {
        return orgMngService.upsert(saveVO);
    }

    /**
     * 删除组织
     *
     * @param id 组织ID
     * @return 删除结果
     */
    @ApiOperation(value = "删除组织")
    @ApiOperationSupport(order = 4)
    @ApiImplicitParam(name = "id", value = "组织ID", required = true, dataType = "int")
    @DeleteMapping(value = "/del/{id}")
    @BusinessObjectOperation(operationType = OperationTypeEnum.DELETE, operationDescription = "删除组织"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> delete(@PathVariable("id") Long id) {
        return orgMngService.delete(id);
    }

    /**
     * 分页查询组织
     *
     * @param queryVO 查询参数
     * @return 组织列表
     */
    @ApiOperation(value = "分页查询组织")
    @ApiOperationSupport(order = 5)
    @PostMapping(value = "/page")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT_PAGE, operationDescription = "分页查询组织"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<PagingVO<OrgPagedRespVO>> queryByPage(@RequestBody OrgPageQueryVO queryVO) {
        return orgMngService.queryByPage(queryVO);
    }

    /**
     * 切换启用禁用
     *
     * @param id 组织ID
     * @return 切换结果
     */
    @PatchMapping(value = "/switch/{id}/enabled")
    @ApiOperation(value = "切换启用禁用")
    @ApiOperationSupport(order = 6)
    @ApiImplicitParam(name = "id", value = "组织ID", required = true)
    @BusinessObjectOperation(operationType = OperationTypeEnum.UPDATE, operationDescription = "切换启用禁用"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> switchEnabled(@PathVariable("id") Long id) {
        return orgMngService.updateEnabled(id);
    }

    /**
     * 重构行政组织域树
     *
     * @param rootOrgId 根节点ID
     * @return 结果
     */
    @PutMapping(value = "/tree/rebuild")
    @ApiOperation(value = "重构行政组织域树")
    @ApiOperationSupport(order = 7)
    @ApiImplicitParam(name = "rootOrgId", value = "根节点ID", paramType = "query")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "重构行政组织域树"
    )
    public ApiResult<Boolean> treeRebuild(@RequestParam(value = "rootOrgId", required = false) Long rootOrgId) {
        return orgMngService.updateTreeRebuild(rootOrgId);
    }
}
