package com.elitescloud.cloudt.system.controller.mng.org;

import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.model.vo.query.org.TenantOrgPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.TenantOrgDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.TenantOrgRespVO;
import com.elitescloud.cloudt.system.model.vo.save.org.OrgTenantSaveVO;
import com.elitescloud.cloudt.system.service.TenantOrgMngService;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;

/**
 * 租户组织管理.
 *
 * @author Kaiser（wang shao）
 * 2022/9/20
 */
@Api(tags = "租户组织管理")
@RestController
@RequestMapping(value = "/mng/tenantOrg", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
public class OrgTenantMngController {

    private final TenantOrgMngService tenantOrgMngService;

    public OrgTenantMngController(TenantOrgMngService tenantOrgMngService) {
        this.tenantOrgMngService = tenantOrgMngService;
    }

    /**
     * 获取租户组织列表
     *
     * @return 租户组织列表
     */
    @ApiOperation(value = "获取租户组织列表")
    @ApiOperationSupport(order = 1)
    @GetMapping(value = "/list")
    public ApiResult<List<TenantOrgRespVO>> getList() {
        return tenantOrgMngService.getList();
    }

    /**
     * 获取租户组织列表
     *
     * @return 租户组织列表
     */
    @ApiOperation(value = "获取租户组织分页列表")
    @ApiOperationSupport(order = 1)
    @PostMapping(value = "/page")
    public ApiResult<PagingVO<TenantOrgRespVO>> getPage(@RequestBody TenantOrgPageQueryVO queryVO) {
        return tenantOrgMngService.pageQuery(queryVO);
    }

    /**
     * 切换启用禁用
     *
     * @param id 租户组织ID
     * @return 切换结果
     */
    @ApiOperation(value = "切换启用禁用")
    @ApiOperationSupport(order = 2)
    @ApiImplicitParam(name = "id", value = "租户组织ID", required = true, dataType = "int")
    @PatchMapping(value = "/switch/enabled/{id}")
    public ApiResult<Long> switchEnabled(@PathVariable("id") Long id) {
        return tenantOrgMngService.updateEnabled(id);
    }

    /**
     * 保存租户组织
     *
     * @param saveVO 保存信息
     * @return 保存结果
     */
    @ApiOperation(value = "保存租户组织")
    @ApiOperationSupport(order = 3)
    @PostMapping(value = "/save")
    public ApiResult<Long> saveTenantOrg(@RequestBody @Valid OrgTenantSaveVO saveVO) {
        return tenantOrgMngService.saveTenantOrg(saveVO);
    }

    /**
     * 获取详细信息
     *
     * @param id 记录ID
     * @return 详细信息
     */
    @ApiOperation(value = "获取详细信息")
    @ApiOperationSupport(order = 4)
    @GetMapping(value = "/get/{id}")
    public ApiResult<TenantOrgDetailRespVO> getDetail(@PathVariable("id") Long id) {
        return tenantOrgMngService.getDetail(id);
    }

    /**
     * 删除租户组织
     *
     * @param id 租户组织ID
     * @return 删除结果
     */
    @ApiOperation(value = "删除租户组织")
    @ApiOperationSupport(order = 5)
    @ApiImplicitParam(name = "id", value = "租户组织ID", required = true, dataType = "int")
    @DeleteMapping(value = "/del/{id}")
    public ApiResult<Long> delete(@PathVariable("id") Long id) {
        return tenantOrgMngService.delete(id);
    }

    /**
     * 分配租户组织管理员
     *
     * @param id      租户组织ID
     * @param adminId 管理员ID
     * @return 分配结果
     */
    @ApiOperation(value = "分配租户组织管理员")
    @ApiOperationSupport(order = 6)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "租户组织ID", required = true, dataType = "int"),
            @ApiImplicitParam(name = "adminId", value = "管理员ID", required = true, dataType = "int"),
    })
    @PatchMapping(value = "/assign/admin/{id}/{adminId}")
    public ApiResult<Long> assignAdmin(@PathVariable("id") Long id, @PathVariable("adminId") Long adminId) {
        return tenantOrgMngService.updateAdmin(id, adminId);
    }
}
