package com.elitescloud.cloudt.system.controller.mng.role;

import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.log.common.BusinessObject;
import com.elitescloud.boot.log.common.BusinessObjectOperation;

import com.elitescloud.boot.log.common.OperationTypeEnum;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.model.vo.resp.api.SysMenuApiRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.dpr.SysDprRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.permission.DataPermissionTreeNodeRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.role.PermissionDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.role.PermissionMenuRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.role.RolePermissionRespVO;
import com.elitescloud.cloudt.system.model.vo.save.dpr.SysDprBatchSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.dpr.SysDprSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.role.MenuApiSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.role.RoleAppPermissionSaveVO;
import com.elitescloud.cloudt.system.service.PermissionMngService;
import com.elitescloud.cloudt.system.service.PermissionQueryService;
import com.elitescloud.cloudt.system.service.common.constant.BusinessObjectConstant;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.List;

/**
 * 权限管理.
 *
 * @author Kaiser（wang shao）
 * 2023/1/10
 */
@Api(tags = "权限管理")
@RestController
@RequestMapping(value = "/mng/permission", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
@BusinessObject(businessType = BusinessObjectConstant.SYS_ROLE_PERMISSION)
public class PermissionMngController {

    private final PermissionQueryService permissionQueryService;
    private final PermissionMngService permissionMngService;

    public PermissionMngController(PermissionQueryService permissionQueryService, PermissionMngService permissionMngService) {
        this.permissionQueryService = permissionQueryService;
        this.permissionMngService = permissionMngService;
    }

    /**
     * 根据权限编码获取权限信息
     *
     * @param code 权限编码
     * @return 权限信息
     */
    @GetMapping(value = "/{code}/detail")
    @ApiOperation(value = "根据权限编码获取权限信息")
    @ApiOperationSupport(order = 1)
    @ApiImplicitParam(name = "code", value = "权限编码")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "根据权限编码获取权限信息"
            , logEnabled = false, resultJsonEnabled = false, argsJsonEnabled = false)
    public ApiResult<PermissionDetailRespVO> permissionDetail(@PathVariable("code") String code) {
        return permissionQueryService.getPermissionDetailByCode(code);
    }

    /**
     * 获取角色的菜单权限
     *
     * @return 菜单权限树
     */
    @ApiOperation(value = "获取角色的菜单权限")
    @ApiOperationSupport(order = 11)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "roleId", value = "角色ID", dataType = "long", required = true),
            @ApiImplicitParam(name = "appCode", value = "应用编码", required = true),
            @ApiImplicitParam(name = "tree", value = "是否返回树形结构数据", dataType = "boolean"),
    })
    @GetMapping(value = "/role/menu")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "获取角色的菜单权限"
            , logEnabled = false, resultJsonEnabled = false, argsJsonEnabled = false)
    public ApiResult<List<RolePermissionRespVO>> getPermissionMenu(@RequestParam(name = "roleId") @NotNull(message = "角色ID为空") Long roleId,
                                                                   @RequestParam(name = "appCode") @NotBlank(message = "应用编码为空") String appCode,
                                                                   @RequestParam(name = "tree", defaultValue = "true") Boolean tree) {
        return permissionQueryService.getPermissionMenuByRole(roleId, appCode, tree);
    }

    /**
     * 保存角色的菜单权限
     *
     * @param roleId     角色ID
     * @param saveVOList 权限信息
     * @return 保存结果
     */
    @ApiOperation(value = "保存角色的菜单权限")
    @ApiOperationSupport(order = 12)
    @ApiImplicitParam(name = "roleId", value = "角色ID", dataType = "int", required = true)
    @PostMapping(value = "/role/{roleId}/menu")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SAVE, operationDescription = "保存角色的菜单权限"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> savePermissionMenu(@PathVariable("roleId") Long roleId,
                                              @RequestBody @Valid List<RoleAppPermissionSaveVO> saveVOList) {
        return permissionMngService.savePermissionMenu(roleId, saveVOList);
    }

    /**
     * 数据权限-数据范围
     *
     * @return 数据范围列表
     */
    @ApiOperation(value = "数据权限-数据范围")
    @ApiOperationSupport(order = 21)
    @GetMapping(value = "/data/ranges")
    public ApiResult<List<CodeNameParam>> getDprDataRanges() {
        return permissionQueryService.dataPermissionRanges();
    }

    /**
     * 数据权限-规则条件
     *
     * @return 规则条件列表
     */
    @ApiOperation(value = "数据权限-规则条件")
    @ApiOperationSupport(order = 22)
    @GetMapping(value = "/data/conditions")
    public ApiResult<List<CodeNameParam>> getDprRuleCondition() {
        return permissionQueryService.dataPermissionRuleCondition();
    }

    /**
     * 数据权限-规则的取值方式
     *
     * @return 规则的取值方式列表
     */
    @ApiOperation(value = "数据权限-规则的取值方式")
    @ApiOperationSupport(order = 23)
    @GetMapping(value = "/data/ruleValueMethod")
    public ApiResult<List<CodeNameParam>> getDprRuleValueMethod() {
        return permissionQueryService.dataPermissionRuleValueMethod();
    }

    /**
     * 获取菜单下的API
     *
     * @param menuCode 菜单编码
     * @return API列表
     */
    @ApiOperation(value = "获取菜单下挂载的API")
    @ApiOperationSupport(order = 31)
    @ApiImplicitParam(name = "menuCode", value = "菜单编码", required = true)
    @GetMapping(value = "/data/menu/api")
    public ApiResult<List<SysMenuApiRespVO>> getApi(@RequestParam(name = "menuCode") @NotBlank(message = "菜单编码为空") String menuCode) {
        return permissionQueryService.getApi(menuCode);
    }

    /**
     * 保存数据权限
     *
     * @param saveVO 保存信息
     * @return 角色ID
     */
    @ApiOperation(value = "保存角色分配的数据权限")
    @ApiOperationSupport(order = 32)
    @PostMapping(value = "/data/roleMenuApi")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SAVE, operationDescription = "保存角色分配的数据权限"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> saveApiDataPermission(@RequestBody @Valid SysDprSaveVO saveVO) {
        return permissionMngService.saveApiDataPermission(saveVO);
    }

    /**
     * 获取角色分配的数据权限
     *
     * @param roleId   角色ID
     * @param menuCode 菜单编码
     * @param apiCode  API接口编码
     * @return 数据权限信息
     */
    @ApiOperation(value = "获取角色分配的数据权限")
    @ApiOperationSupport(order = 33)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "roleId", value = "角色ID", required = true),
            @ApiImplicitParam(name = "menuCode", value = "菜单编码", required = true),
            @ApiImplicitParam(name = "apiCode", value = "接口编码", required = true),
    })
    @GetMapping(value = "/data/roleMenuApi")
    public ApiResult<SysDprRespVO> getApiDataPermission(@RequestParam(name = "roleId") @NotNull(message = "角色ID为空") Long roleId,
                                                        @RequestParam(name = "menuCode") @NotBlank(message = "菜单编码为空") String menuCode,
                                                        @RequestParam(name = "apiCode") @NotBlank(message = "接口编码为空") String apiCode) {
        return permissionQueryService.getApiDataPermission(roleId, menuCode, apiCode);
    }

    /**
     * 保存菜单挂载的接口
     *
     * @param saveVO 保存信息
     * @return 结果
     */

    @ApiOperation(value = "保存接口绑定的菜单")
    @ApiOperationSupport(order = 34)
    @PostMapping(value = "/api/menus")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SAVE, operationDescription = "保存菜单挂载的接口"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Boolean> saveMenuApi(@RequestBody @Valid MenuApiSaveVO saveVO) {
        return permissionMngService.saveMenuApi(saveVO);
    }

    @ApiOperation(value = "获取接口绑定的菜单")
    @ApiOperationSupport(order = 34)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "apiId", value = "接口ID", required = true),
            @ApiImplicitParam(name = "appCode", value = "应用编码"),
    })
    @GetMapping(value = "/api/menus")
    public ApiResult<List<Long>> getMenuApi(@RequestParam(name = "apiId") @NotNull(message = "接口ID为空") Long apiId,
                                            @RequestParam(name = "appCode", required = false) String appCode) {
        return permissionQueryService.getMenuIdOfApi(apiId, appCode);
    }

    @ApiOperation(value = "获取接口绑定的菜单详细信息")
    @ApiOperationSupport(order = 34)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "apiId", value = "接口ID", required = true),
            @ApiImplicitParam(name = "appCode", value = "应用编码"),
    })
    @GetMapping(value = "/api/menusDetail")
    public ApiResult<List<PermissionMenuRespVO>> getMenuApiDetail(@RequestParam(name = "apiId") @NotNull(message = "接口ID为空") Long apiId,
                                                                  @RequestParam(name = "appCode", required = false) String appCode) {
        return permissionQueryService.getMenuDetailOfApi(apiId, appCode);
    }

    @ApiOperation(value = "获取分配数据权限的菜单接口树")
    @ApiOperationSupport(order = 41)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "roleId", value = "角色ID", required = true, dataType = "int"),
            @ApiImplicitParam(name = "tree", value = "是否返回树状数据", dataType = "boolean"),
    })
    @GetMapping(value = "/dataPermission/tree")
    public ApiResult<List<DataPermissionTreeNodeRespVO>> getDataPermissionTree(@RequestParam(name = "roleId") @NotNull(message = "角色ID为空") Long roleId,
                                                                               @RequestParam(name = "tree", required = false, defaultValue = "false") Boolean tree) {
        return permissionQueryService.dataPermissionTree(roleId, null, tree);
    }

    /**
     * 保存角色分配的批量数据权限
     *
     * @param saveVO 保存信息
     * @return 角色ID
     */
    @ApiOperation(value = "保存角色分配的批量数据权限")
    @ApiOperationSupport(order = 42)
    @PostMapping(value = "/dataPermission/roleMenuApiBatch")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SAVE, operationDescription = "保存角色分配的批量数据权限"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Boolean> saveApiDataPermission(@RequestBody @Valid SysDprBatchSaveVO saveVO) {
        return permissionMngService.saveApiDataPermission(saveVO);
    }
}
