package com.elitescloud.cloudt.system.controller.mng.user;

import com.elitescloud.boot.auth.client.common.InterceptUri;
import com.elitescloud.boot.log.common.BusinessObject;
import com.elitescloud.boot.log.common.BusinessObjectOperation;
import com.elitescloud.boot.log.common.OperationTypeEnum;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.model.vo.query.user.UserPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.user.UserDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.user.UserPageRespVO;
import com.elitescloud.cloudt.system.model.vo.save.user.UserSaveVO;
import com.elitescloud.cloudt.system.service.UserMngService;
import com.elitescloud.cloudt.system.service.common.constant.BusinessObjectConstant;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;
import java.util.Set;

/**
 * 账号管理.
 *
 * @author Kaiser（wang shao）
 * 2022/9/20
 */
@Api(tags = "账号管理")
@RestController
@RequestMapping(value = "/mng/user", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
@BusinessObject(businessType = BusinessObjectConstant.SYS_USER)
public class UserMngController {
    static {
        InterceptUri.addAllowUri(Set.of("/mng/user/cas/clear/userId"));
    }

    private final UserMngService userMngService;

    public UserMngController(UserMngService userMngService) {
        this.userMngService = userMngService;
    }

    /**
     * 分页查询账号
     *
     * @param queryVO 查询参数
     * @return 分页数据
     */
    @ApiOperation(value = "分页查询账号")
    @ApiOperationSupport(order = 1)
    @PostMapping(value = "/page")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT_PAGE, operationDescription = "分页查询账号"
            , logEnabled = false, resultJsonEnabled = false, argsJsonEnabled = false)
    public ApiResult<PagingVO<UserPageRespVO>> page(@RequestBody UserPageQueryVO queryVO) {
        return userMngService.page(queryVO);
    }

    /**
     * 切换启用禁用
     *
     * @param id 账号ID
     * @return 切换结果
     */
    @ApiOperation(value = "切换启用禁用")
    @ApiOperationSupport(order = 2)
    @ApiImplicitParam(name = "id", value = "账号ID", required = true, dataType = "int")
    @PatchMapping(value = "/switch/enabled/{id}")
    @BusinessObjectOperation(operationType = OperationTypeEnum.UPDATE, operationDescription = "切换启用禁用"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> switchEnabled(@PathVariable("id") Long id) {
        return userMngService.updateEnabled(id);
    }

    /**
     * 删除账号
     *
     * @param id 账号ID
     * @return 删除结果
     */
    @ApiOperation(value = "删除账号")
    @ApiOperationSupport(order = 3)
    @ApiImplicitParam(name = "id", value = "账号ID", required = true, dataType = "int")
    @DeleteMapping(value = "/del/{id}")
    @BusinessObjectOperation(operationType = OperationTypeEnum.DELETE, operationDescription = "删除账号"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> delete(@PathVariable("id") Long id) {
        return userMngService.delete(id);
    }

    /**
     * 获取账号详情
     *
     * @param id 账号ID
     * @return 账号信息
     */
    @ApiOperation(value = "获取账号详情")
    @ApiOperationSupport(order = 4)
    @ApiImplicitParam(name = "id", value = "账号ID", required = true, dataType = "int")
    @GetMapping(value = "/get/{id}")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "获取账号详情"
            , logEnabled = false, resultJsonEnabled = false, argsJsonEnabled = false)
    public ApiResult<UserDetailRespVO> get(@PathVariable("id") Long id) {
        return userMngService.get(id);
    }

    /**
     * 新增(修改)账户信息
     *
     * @param saveVO 账户信息
     * @return 保存结果
     */
    @ApiOperation(value = "新增(修改)账户信息")
    @ApiOperationSupport(order = 8)
    @PutMapping(value = "/upsert")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SAVE, operationDescription = "新增(修改)账户信息"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> upsert(@RequestBody @Valid UserSaveVO saveVO) {
        return userMngService.save(saveVO);
    }

    /**
     * 重置密码
     *
     * @param id 账号ID
     * @return 新增结果
     */
    @ApiOperation(value = "重置密码")
    @ApiOperationSupport(order = 9)
    @PatchMapping(value = "/reset/{id}/password")
    @BusinessObjectOperation(operationType = OperationTypeEnum.UPDATE, operationDescription = "重置密码"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> updatePassword(@PathVariable("id") Long id) {
        return userMngService.updatePassword(id);
    }

    /**
     * 清除CAS的UserId
     *
     * @param ids 账号ID
     * @return 结果
     */
    @ApiOperation(value = "清除CAS的UserId")
    @ApiOperationSupport(order = 10)
    @PatchMapping(value = "/cas/clear/userId")
    @BusinessObjectOperation(operationType = OperationTypeEnum.ELSE, operationDescription = "清除CAS的UserId"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Boolean> clearCasUserId(@RequestBody List<Long> ids) {
        return userMngService.clearCasUserId(ids);
    }
}
