package com.elitescloud.cloudt.system.factory.seq;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.constant.NextValuePeriod;
import com.elitescloud.cloudt.platform.model.entity.SysPlatformNextNumberDO;
import com.elitescloud.cloudt.platform.service.repo.number.SysPlatformNextNumberRepo;
import com.elitescloud.cloudt.system.model.bo.SysSeqNextNumberBO;
import com.elitescloud.cloudt.system.service.repo.SeqNextNumRepoProc;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.time.LocalDateTime;
import java.time.temporal.WeekFields;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 发号服务.
 *
 * @author Kaiser（wang shao）
 * 2022/11/29
 */
@Component
@Log4j2
class NextNumGenerateService {
    /**
     * 重试次数
     */
    private static final int REPEAT_TIMES = 6;

    @Autowired
    private SeqNextNumRepoProc nextNumRepoProc;
    @Autowired
    private SysPlatformNextNumberRepo nextNumberRepo;

    /**
     * 创建下一序号
     * <p>
     * 初始化创建下一序号
     *
     * @param nnCode 序号编码
     * @return 序号ID
     */
    @Transactional(rollbackFor = Exception.class, propagation = Propagation.REQUIRES_NEW)
    public Long createNextNumberForInit(@NotBlank String appCode, @NotBlank String nnCode) {
        var nextNumberDO = new SysPlatformNextNumberDO();
        nextNumberDO.setAppCode(appCode);
        nextNumberDO.setCode(nnCode);
        nextNumberDO.setName("递增序列");
        nextNumberDO.setStep(1);
        nextNumberDO.setNextNumber(1L);
//        nextNumberDO.setNnPeriod("D");
        nextNumberDO.setNnTime(LocalDateTime.now());
        nextNumberDO.setVersion(1);
        nextNumberDO.setEnabled(true);
        nextNumberDO.setInternal(false);

        nextNumberRepo.save(nextNumberDO);

        return nextNumberDO.getId();
    }

    /**
     * 生成下一序号的值
     *
     * @param nnId  下一序号的ID
     * @param nnLen 下一序号的长度
     * @return 下一序号的值
     */
    @Transactional(rollbackFor = Exception.class, propagation = Propagation.REQUIRES_NEW)
    public Long nextNumber(@NotNull Long nnId, Integer nnLen) {
        var nextNumberBO = nextNumRepoProc.getBo(nnId);
        if (nextNumberBO == null) {
            throw new BusinessException("未找到下一序号[" + nnId + "]");
        }

        AtomicReference<SysSeqNextNumberBO> nextNumberAtomic = new AtomicReference<>(nextNumberBO);
        return buildNextNumber(nextNumberAtomic, nnLen, 1);
    }

    /**
     * 生成下一序号的值
     *
     * @param nnId  下一序号的ID
     * @param nnLen 下一序号的长度
     * @param num   生成序号的数量
     * @return 下一序号的值
     */
    @Transactional(rollbackFor = Exception.class, propagation = Propagation.REQUIRES_NEW)
    public List<Long> nextNumber(@NotNull Long nnId, Integer nnLen, int num) {
        var nextNumberBO = nextNumRepoProc.getBo(nnId);
        if (nextNumberBO == null) {
            throw new BusinessException("未找到下一序号[" + nnId + "]");
        }

        AtomicReference<SysSeqNextNumberBO> nextNumberAtomic = new AtomicReference<>(nextNumberBO);
        Long min = buildNextNumber(nextNumberAtomic, nnLen, num);
        if (min == null) {
            // 生成失败
            return Collections.emptyList();
        }
        long max = Objects.requireNonNull(nextNumberAtomic.get().getNextNumber(), "生成下一序号异常");
        return Stream.iterate(min, t -> t < max, t -> t + nextNumberAtomic.get().getStep()).collect(Collectors.toList());
    }

    private Long buildNextNumber(AtomicReference<SysSeqNextNumberBO> nextNumberAtomic, Integer nnLen, int num) {
        // 生成下一编号
        var nowTime = LocalDateTime.now();
        SysSeqNextNumberBO nextNumberBO = nextNumberAtomic.get();

        int i = 0;
        while (i <= REPEAT_TIMES) {
            var nextNumber = generateNextNumber(nextNumberBO, nnLen, nowTime);

            long theNextNum = nextNumber + (long) (nextNumberBO.getStep() == null || nextNumberBO.getStep() <= 0 ? 1 : nextNumberBO.getStep()) * num;

            var success = nextNumRepoProc.updateValue(nextNumberBO.getId(), theNextNum, nowTime, nextNumberBO.getVersion());
            if (success) {
                log.info("{}成功生成下一序号：{}，重试次数：{}", nextNumberBO.getCode(), nextNumber, i);
                nextNumberBO.setNextNumber(theNextNum);
                nextNumberAtomic.set(nextNumberBO);
                return nextNumber;
            }
            log.info("{}失败生成下一序号：{}，重试第次：{}", nextNumberBO.getCode(), nextNumber, i);
            i++;
            nextNumberBO = nextNumRepoProc.getBo(nextNumberBO.getId());
        }

        log.warn("{}生成下一序号失败", nextNumberBO.getCode());
        return null;
    }

    private Long generateNextNumber(SysSeqNextNumberBO nextNumberBO, Integer nnLen, LocalDateTime nowTime) {
        if (CharSequenceUtil.isNotBlank(nextNumberBO.getNnPeriod())) {
            if (nextNumberBO.getNnTime() != null) {
                NextValuePeriod period = NextValuePeriod.parse(nextNumberBO.getNnPeriod());
                if (period == null) {
                    return nextNumberBO.getNextNumber();
                }
                // 是否同年
                var isSameYear = nowTime.getYear() == nextNumberBO.getNnTime().getYear();
                switch (period) {
                    case D:
                        var nowD = nowTime.getDayOfYear();
                        var nnD = nextNumberBO.getNnTime().getDayOfYear();
                        if (!isSameYear || nowD != nnD) {
                            return 1L;
                        }
                        return nextNumberBO.getNextNumber();
                    case W:
                        if (!isSameYear
                                || (nowTime.get(WeekFields.ISO.weekOfWeekBasedYear())) != nextNumberBO.getNnTime().get(WeekFields.ISO.weekOfWeekBasedYear())) {
                            return 1L;
                        }
                        return nextNumberBO.getNextNumber();
                    case M:
                        if (!isSameYear || nowTime.getMonthValue() != nextNumberBO.getNnTime().getMonthValue()) {
                            return 1L;
                        }
                        return nextNumberBO.getNextNumber();
                    case Y:
                        if (!isSameYear) {
                            return 1L;
                        }
                        return nextNumberBO.getNextNumber();
                    default:
                        return nextNumberBO.getNextNumber();
                }
            }
        }
        if (nnLen != null && nextNumberBO.getNextNumber() >= Math.pow(10, nnLen)) {
            return 1L;
        }
        return nextNumberBO.getNextNumber();
    }
}
