package com.elitescloud.cloudt.system.modules.message.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.util.RestTemplateFactory;
import com.elitescloud.cloudt.messenger.Messenger;
import com.elitescloud.cloudt.messenger.message.SmsMessageVO;
import com.elitescloud.cloudt.messenger.model.FileMessageAttachmentVO;
import com.elitescloud.cloudt.messenger.model.MessageAccountVO;
import com.elitescloud.cloudt.messenger.model.MessageAttachmentVO;
import com.elitescloud.cloudt.system.constant.MsgSendStateEnum;
import com.elitescloud.cloudt.system.constant.MsgSendTypeEnum;
import com.elitescloud.cloudt.system.model.vo.sbean.EmployeePagedRespBean;
import com.elitescloud.cloudt.system.modules.message.repository.SysMsgSendRecordDtlRepository;
import com.elitescloud.cloudt.system.modules.message.vo.respose.SysMsgTemplateConfigVO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author : chen.niu
 * @description : 消息发送实现方法
 * @date : 2023/10/30 14:27
 */
@Service
@Slf4j
public class SysMsgSendService extends SysMsgSendCommonServiceAbstract {


    private final SysMsgSendRecordDtlRepository sysMsgSendRecordDtlRepository;
    private final RestTemplate restTemplate = RestTemplateFactory.instance();

    public SysMsgSendService(SysMsgSendRecordDtlRepository sysMsgSendRecordDtlRepository) {
        this.sysMsgSendRecordDtlRepository = sysMsgSendRecordDtlRepository;

    }

    /**
     * 批量发送微信BOOT消息
     **/
    public void sendBatchWxBootMsg(String uuidBatch, List<MessageAccountVO> sendPathMap,
                                   MsgSendTypeEnum sendTypeEnum,
                                   SysMsgTemplateConfigVO msgTemplateTxtReplaceDTO,
                                   List<EmployeePagedRespBean> carbonUserEmployeeList,
                                   List<String> fileCodes) {

        String MSG_TYPE = "markdown";
        String wxBootUrl = msgTemplateTxtReplaceDTO.getExternalTemplateId();
        String title = msgTemplateTxtReplaceDTO.getMesTitle();
        String mesText = msgTemplateTxtReplaceDTO.getMesText();
        String content = "# " + title + " \n " + mesText;
        Map<String, Object> body = new HashMap<>();
        body.put("msgtype", MSG_TYPE);
        body.put(MSG_TYPE, Map.of("content", content));
        log.info("【企业微信】发送预警：{}, {}", wxBootUrl, content);

        ResponseEntity<String> resp = null;
        try {
            resp = restTemplate.exchange(wxBootUrl, HttpMethod.POST, new HttpEntity<>(body), String.class);
            if (resp.getStatusCode().is2xxSuccessful()) {
                log.info("发送成功！");
                sysMsgSendRecordDtlRepository.updateByCondition(
                        resp.getStatusCode().name(), null, MsgSendStateEnum.OK.name(),
                        LocalDateTime.now(), uuidBatch, sendTypeEnum.name());
                return;
            } else {
                sysMsgSendRecordDtlRepository.updateByCondition(
                        resp.getBody(), null, MsgSendStateEnum.FAILED.name(),
                        LocalDateTime.now(), uuidBatch, sendTypeEnum.name());
            }
        } catch (RestClientException e) {
            log.error("发送预警异常：", e);
            sysMsgSendRecordDtlRepository.updateByCondition(
                    e.getMessage(), null, MsgSendStateEnum.FAILED.name(),
                    LocalDateTime.now(), uuidBatch, sendTypeEnum.name());
            return;
        }
        log.info("发送失败：{}, {}", resp.getStatusCodeValue(), resp.getBody());

    }

    /**
     * 批量发送站内信
     **/
    public void sendBatchSiteMsg(String uuidBatch,
                                 List<MessageAccountVO> accountList,
                                 MsgSendTypeEnum sendTypeEnum,
                                 SysMsgTemplateConfigVO msgTemplateTxtReplaceDTO) {

        if (CollectionUtils.isEmpty(accountList)) {
            log.error("消息服务：没有一个接收人的账号，站内信消息将不会发送！");
        } else {
            try {
                String result = Messenger.siteMessage()
                        .setReceiverList(accountList)
                        .setSubject(msgTemplateTxtReplaceDTO.getMesTitle())
                        .setContent(msgTemplateTxtReplaceDTO.getMesText())
                        .setBusinessType(CharSequenceUtil.blankToDefault(msgTemplateTxtReplaceDTO.getBusinessType(), "default"))
                        .setBusinessParams(msgTemplateTxtReplaceDTO.getBusinessData())
                        .build()
                        .send();
                sysMsgSendRecordDtlRepository.updateByCondition(
                        result, null, MsgSendStateEnum.OK.name(),
                        LocalDateTime.now(), uuidBatch, sendTypeEnum.name()
                );
            } catch (Exception e) {
                log.error("发送站内信异常：{}", uuidBatch, e);
                sysMsgSendRecordDtlRepository.updateByCondition(
                        null, e.getMessage(), MsgSendStateEnum.FAILED.name(),
                        LocalDateTime.now(), uuidBatch, sendTypeEnum.name()
                );
            }
        }
    }

    /**
     * 批量发送App消息
     **/
    public void sendBatchAppMsg(String uuidBatch,
                                List<MessageAccountVO> accountList,
                                MsgSendTypeEnum sendTypeEnum,
                                SysMsgTemplateConfigVO msgTemplateTxtReplaceDTO) {

        if (CollectionUtils.isEmpty(accountList)) {
            log.error("消息服务：没有一个接收人的账号，App消息将不会发送！");
        } else {
            try {
                String result = Messenger.appMessage()
                        .setReceiverList(accountList)
                        .setSubject(msgTemplateTxtReplaceDTO.getMesTitle())
                        .setContent(msgTemplateTxtReplaceDTO.getMesText())
                        .setBusinessType(CharSequenceUtil.blankToDefault(msgTemplateTxtReplaceDTO.getBusinessType(), "default"))
                        .setBusinessParams(msgTemplateTxtReplaceDTO.getBusinessData())
                        .build()
                        .send();
                sysMsgSendRecordDtlRepository.updateByCondition(
                        result, null, MsgSendStateEnum.OK.name(),
                        LocalDateTime.now(), uuidBatch, sendTypeEnum.name()
                );
            } catch (Exception e) {
                log.error("发送APP消息异常：{}", uuidBatch, e);
                sysMsgSendRecordDtlRepository.updateByCondition(
                        null, e.getMessage(), MsgSendStateEnum.FAILED.name(),
                        LocalDateTime.now(), uuidBatch, sendTypeEnum.name()
                );
            }
        }
    }

    /**
     * 批量发送短信
     **/
    public void sendBatchSmsMsg(String uuidBatch,
                                List<MessageAccountVO> sendEmailList,
                                MsgSendTypeEnum sendTypeEnum,
                                SysMsgTemplateConfigVO msgTemplateTxtReplaceDTO, Map<String, String> templateParams) {
        if (CollectionUtils.isEmpty(sendEmailList)) {
            log.error("消息服务：没有一个接收人的手机号，短信消息将不会发送！");
            return;
        }

        SmsMessageVO messageVO = null;
        if (StringUtils.hasText(msgTemplateTxtReplaceDTO.getExternalTemplateId())) {
            messageVO = Messenger.templateSms()
                    .setTemplateCode(msgTemplateTxtReplaceDTO.getExternalTemplateId())
                    .setSignName(CharSequenceUtil.blankToDefault(msgTemplateTxtReplaceDTO.getExternalSignName(), null))
                    .setTemplateParam(templateParams)
                    .setReceiverList(sendEmailList)
                    .setSubject(msgTemplateTxtReplaceDTO.getMesTitle())
                    .setContent(msgTemplateTxtReplaceDTO.getMesText())
                    .build();
        } else {
            messageVO = Messenger.sms()
                    .setReceiverList(sendEmailList)
                    .setSubject(msgTemplateTxtReplaceDTO.getMesTitle())
                    .setContent(msgTemplateTxtReplaceDTO.getMesText())
                    .setSignName(CharSequenceUtil.blankToDefault(msgTemplateTxtReplaceDTO.getExternalSignName(), null))
                    .build();
        }

        try {
            String result = messageVO.send();
            sysMsgSendRecordDtlRepository.updateByCondition(
                    result, null, MsgSendStateEnum.OK.name(),
                    LocalDateTime.now(), uuidBatch, sendTypeEnum.name()
            );
        } catch (Exception e) {
            log.error("发送短信失败：{}", uuidBatch, e);
            sysMsgSendRecordDtlRepository.updateByCondition(
                    null, e.getMessage(), MsgSendStateEnum.FAILED.name(),
                    LocalDateTime.now(), uuidBatch, sendTypeEnum.name()
            );
        }
    }

    /**
     * 批量发送邮件
     * uuidBatch批次号
     * sendEmailList 接收人
     * carbonUserEmployeeList 抄送人
     * fileCodes 附件编码
     *
     **/
    public void sendBatchEmailMsg(String uuidBatch,
                                  List<MessageAccountVO> sendEmailList,
                                  MsgSendTypeEnum sendTypeEnum,
                                  SysMsgTemplateConfigVO msgTemplateTxtReplaceDTO,
                                  List<EmployeePagedRespBean> carbonUserEmployeeList,
                                  List<String> fileCodes) {
        if (CollectionUtils.isEmpty(sendEmailList)) {
            log.error("消息服务：没有一个邮件接收人的邮箱，邮件消息将不会发送！");
        } else {
            try {
                String result = null;
                List<MessageAccountVO> carbonUserEmployeeEmailList = new ArrayList<>();
                if (carbonUserEmployeeList != null) {
                    carbonUserEmployeeEmailList = carbonUserEmployeeList.stream().map(
                            employeePagedRespBean -> {
                                //参考 addSendPathMapByEmployeeSendType方法设置规则
                                if (StringUtils.hasText(employeePagedRespBean.getEmail())) {
                                    MessageAccountVO messageAccountVO = new MessageAccountVO();
                                    messageAccountVO.setAccount(employeePagedRespBean.getEmail());
                                    messageAccountVO.setNickName(employeePagedRespBean.getLastName());
                                    return messageAccountVO;
                                } else {
                                    return null;
                                }
                            }
                    ).collect(Collectors.toList());
                }
                carbonUserEmployeeEmailList.removeIf(Objects::isNull);

                List<MessageAttachmentVO> messageAttachmentVOS = new ArrayList<>();
                if (fileCodes != null && !fileCodes.isEmpty()) {
                    messageAttachmentVOS = fileCodes.stream().map(s -> {
                        FileMessageAttachmentVO fileMessageAttachmentVO = new FileMessageAttachmentVO();
                        fileMessageAttachmentVO.setFileCode(s);
                        return fileMessageAttachmentVO;
                    }).collect(Collectors.toList());
                }

                result = Messenger.email()
                        .setSubject(msgTemplateTxtReplaceDTO.getMesTitle())
                        .setContent(msgTemplateTxtReplaceDTO.getMesText())
                        .setReceiverList(sendEmailList)
                        .setCcList(carbonUserEmployeeEmailList)
                        .setAttachments(messageAttachmentVOS)
                        .setRetryableOnFailure(true)
                        .build()
                        .send();
                //更新成功状态
                sysMsgSendRecordDtlRepository.updateByCondition(
                        result, null, MsgSendStateEnum.OK.name(),
                        LocalDateTime.now(), uuidBatch, sendTypeEnum.name()
                );
            } catch (Exception e) {
                //更新失败状态
                log.error("发送邮件失败：{}", uuidBatch, e);
                sysMsgSendRecordDtlRepository.updateByCondition(
                        null, e.getMessage(), MsgSendStateEnum.FAILED.name(),
                        LocalDateTime.now(), uuidBatch, sendTypeEnum.name()
                );
            }
        }
    }


}
