package com.elitescloud.cloudt.system.modules.message.service.impl;

/**
 * @author : chen.niu
 * @description :
 * @date : 2023/5/23 12:37
 */

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.common.param.IdNameParam;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.common.MsgTemplateReceiverEnum;
import com.elitescloud.cloudt.system.constant.MsgSendTypeEnum;
import com.elitescloud.cloudt.system.modules.message.bo.MsgTemplateReceiverBO;
import com.elitescloud.cloudt.system.modules.message.convert.SysMsgTemplateConfigConvert;
import com.elitescloud.cloudt.system.modules.message.convert.SysMsgTemplateConvert;
import com.elitescloud.cloudt.system.modules.message.entity.*;
import com.elitescloud.cloudt.system.modules.message.repository.*;
import com.elitescloud.cloudt.system.modules.message.service.SysMsgTemplateService;
import com.elitescloud.cloudt.system.modules.message.vo.request.*;
import com.elitescloud.cloudt.system.modules.message.vo.respose.SysMsgTemplateConfigVO;
import com.elitescloud.cloudt.system.modules.message.vo.respose.SysMsgTemplatePageRespVO;
import com.elitescloud.cloudt.system.modules.message.vo.respose.SysMsgTemplateRespVO;
import com.elitescloud.cloudt.system.service.EmployeeQueryService;
import com.elitescloud.cloudt.system.service.RoleQueryService;
import com.elitescloud.cloudt.system.service.UserQueryService;
import com.elitescloud.cloudt.system.service.repo.EmployeeRepo;
import com.elitescloud.cloudt.system.service.repo.EmployeeRepoProc;
import com.elitescloud.cloudt.system.service.repo.UserRoleRepoProc;
import com.elitescloud.cloudt.system.vo.SysUdcVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.dsl.Expressions;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service
@AllArgsConstructor
public class SysMsgTemplateServiceImpl implements SysMsgTemplateService {
    private final EmployeeRepo employeeRepo;
    private final EmployeeRepoProc employeeRepoProc;
    private final UserRoleRepoProc userRoleRepoProc;
    private final SysMsgTemplateRepository sysMsgTemplateRepository;
    private final SysMsgTemplateRepoProc sysMsgTemplateRepoProc;
    private final SysMsgTemplateConfigRepository sysMsgTemplateConfigRepository;
    private final SysMsgTemplateConfigRepoProc sysMsgTemplateConfigRepoProc;
    private final SysMsgTemplateReceiverRepoProc sysMsgTemplateReceiverRepoProc;
    private final UserQueryService userQueryService;
    private final RoleQueryService roleQueryService;
    private final EmployeeQueryService employeeQueryService;

    @Override
    public ApiResult<List<SysUdcVO>> getSysMsgSendType() {
        List<SysUdcVO> sysUdcVOList = MsgSendTypeEnum.getSysUdcVOList();
        return ApiResult.ok(sysUdcVOList);
    }

    @Override
    public ApiResult<PagingVO<SysMsgTemplatePageRespVO>> getSysMsgTemplateList(
            SysMsgTemplateQueryVO queryVO) {
        QSysMsgTemplateDO qSysMsgTemplateDO = QSysMsgTemplateDO.sysMsgTemplateDO;
        Predicate predicate = Expressions.booleanTemplate("1=1");
        predicate = org.apache.commons.lang3.StringUtils.isBlank(queryVO.getTemplateName()) ? predicate :
                ExpressionUtils.and(predicate, qSysMsgTemplateDO.templateName.eq(queryVO.getTemplateName()));
        predicate = org.apache.commons.lang3.StringUtils.isBlank(queryVO.getTemplateCode()) ? predicate :
                ExpressionUtils.and(predicate, qSysMsgTemplateDO.templateCode.eq(queryVO.getTemplateCode()));
        var page = sysMsgTemplateRepository.findAll(predicate, queryVO.getPageRequest());

        if (queryVO.getQueryTemplateConfigFlag()) {
            //查询主表里面的全部明细表 批量查询
            var configQDO = QSysMsgTemplateConfigDO.sysMsgTemplateConfigDO;
            List<String> tCodes = page.get().map(SysMsgTemplateDO::getTemplateCode).collect(Collectors.toList());
            List<SysMsgTemplateConfigVO> configVoList = new ArrayList();
            sysMsgTemplateConfigRepository.findAll(configQDO.deleteFlag.eq(0)
                            .and(configQDO.templateCode.in(tCodes)))
                    .forEach(configDo -> {
                        configVoList.add(SysMsgTemplateConfigConvert.INSTANCE.doToVo(configDo));
                    });

            //将明细表数据分组 然后设置到返回的主表vo中
            var configMapGroup = configVoList.stream()
                    .collect(Collectors.groupingBy(SysMsgTemplateConfigVO::getTemplateCode));
            var pageVo = page.get().map(msgTemplateDO -> {
                var vo = SysMsgTemplateConvert.INSTANCE.doToSysMsgTemplatePageRespVO(msgTemplateDO);
                vo.setTemplateConfigs(configMapGroup.get(vo.getTemplateCode()));
                return vo;
            }).collect(Collectors.toList());
            var pagingVo = PagingVO.<SysMsgTemplatePageRespVO>builder()
                    .total(page.getTotalElements())
                    .setRecords(pageVo);
            return ApiResult.ok(pagingVo);
        } else {
            var pagingVo = PagingVO.<SysMsgTemplatePageRespVO>builder()
                    .total(page.getTotalElements())
                    .setRecords(page.get().map(SysMsgTemplateConvert.INSTANCE::doToSysMsgTemplatePageRespVO)
                            .collect(Collectors.toList()));
            return ApiResult.ok(pagingVo);
        }
    }

    @Override
    public ApiResult<SysMsgTemplateRespVO> getSysMsgTemplate(Long id, Boolean queryTemplateConfigFlag) {
        Assert.notNull(id, "ID为空");

        return sysMsgTemplateRepoProc.getOptional(id)
                .map(template -> {
                    var sysMsgTemplateRespVO = SysMsgTemplateConvert.INSTANCE.doToVo(template);
                    if (queryTemplateConfigFlag != null && queryTemplateConfigFlag) {

                        List<SysMsgTemplateConfigVO> configVoList = getTemplateConfigList(template);
                        sysMsgTemplateRespVO.setTemplateConfigs(configVoList);
                    }

                    return sysMsgTemplateRespVO;
                })
                .map(ApiResult::ok).orElseThrow(() -> new BusinessException("数据不存在"));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Long> templateAndConfigSave(TemplateAndConfigSaveVO vo) {
        Assert.hasText(vo.getTemplateCode(), "模板编码为空");
        var existsCode = sysMsgTemplateRepoProc.existsTemplateCode(vo.getTemplateCode());
        if (existsCode) {
            return ApiResult.fail("模板编号不能重复");
        }

        SysMsgTemplateDO sysMsgTemplateDO = new SysMsgTemplateDO();
        sysMsgTemplateDO.setTemplateCode(vo.getTemplateCode());
        sysMsgTemplateDO.setTemplateName(vo.getTemplateName());
        sysMsgTemplateDO.setTemplateGroup(vo.getTemplateGroup());
        sysMsgTemplateDO.setTemplateDescribe(vo.getTemplateDescribe());
        sysMsgTemplateRepoProc.save(sysMsgTemplateDO);
        var id = sysMsgTemplateDO.getId();

        // 保存模板配置项
        sysMsgTemplateConfigSaveAll(vo.getTemplateConfigs(), id, sysMsgTemplateDO.getTemplateCode(), sysMsgTemplateDO.getTemplateName());

        return ApiResult.ok(id);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Long> templateAndConfigUpdate(Long id, TemplateAndConfigUpdateVO vo) {
        Assert.notNull(id, "ID为空");

        sysMsgTemplateRepoProc.getOptional(id).ifPresentOrElse(sysMsgTemplateDO -> {
                    sysMsgTemplateDO.setTemplateName(vo.getTemplateName());
                    sysMsgTemplateDO.setTemplateGroup(vo.getTemplateGroup());
                    sysMsgTemplateDO.setTemplateDescribe(vo.getTemplateDescribe());
                    //修改
                    sysMsgTemplateRepoProc.save(sysMsgTemplateDO);

                    //全删
                    sysMsgTemplateConfigRepository.deleteByTemplateId(id);
                    //全插入
                    sysMsgTemplateConfigSaveAll(vo.getTemplateConfigs(), id, sysMsgTemplateDO.getTemplateCode(), sysMsgTemplateDO.getTemplateName());
                },
                () -> {
                    throw new BusinessException("修改的数据不存在");
                });
        return ApiResult.ok(id);
    }

    @Override
    public ApiResult<Boolean> sendTypeSwitchUpdate(Long id, Boolean flag) {
        sysMsgTemplateConfigRepository.findById(id).ifPresentOrElse(sysMsgTemplateConfigDO -> {
                    sysMsgTemplateConfigDO.setSendTypeSwitch(flag);
                    sysMsgTemplateConfigRepository.save(sysMsgTemplateConfigDO);
                },
                () -> {
                    throw new BusinessException("数据不存在");
                });
        return ApiResult.ok(true);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Boolean> deleteSysMsgTemplate(TemplateDeleteVO templateDeleteVO) {
        val tlist = templateDeleteVO.getTemplateIdList();
        sysMsgTemplateRepository.deleteAllById(tlist);
        tlist.forEach(sysMsgTemplateConfigRepository::deleteByTemplateId);
        return ApiResult.ok(true);
    }

    @Override
    public ApiResult<Map<MsgSendTypeEnum, Set<Long>>> queryDefaultReceivers(String templateCode) {
        Assert.hasText(templateCode, "模板编码为空");
        // 查询模板
        var templateId = sysMsgTemplateRepoProc.getIdByTemplateCode(templateCode);
        if (templateId == null) {
            return ApiResult.fail("未找到模板配置，请确认模板编码正确");
        }

        // 查询启用的模板配置
        var configIdMap = sysMsgTemplateConfigRepoProc.getIdsByTemplateId(templateId, true);
        if (configIdMap.isEmpty()) {
            return ApiResult.fail("消息模板配置均未启用，请先启用消息模板");
        }

        // 查询默认接收人
        Map<MsgSendTypeEnum, Set<Long>> result = new HashMap<>(configIdMap.size());
        for (Map.Entry<String, Long> entry : configIdMap.entrySet()) {
            var sendType = MsgSendTypeEnum.parse(entry.getKey());
            if (sendType == null) {
                log.warn("未知发送类型：{}", entry.getKey());
                continue;
            }

            result.put(sendType, this.getTemplateReceiverUserIds(templateId, entry.getValue()));
        }
        return ApiResult.ok(result);
    }

    /**
     * 私有方法：模板配置批量新增
     ***/
    private void sysMsgTemplateConfigSaveAll(List<TemplateConfigSaveVO> configList, Long tid, String tCode, String tName) {
        sysMsgTemplateReceiverRepoProc.deleteByTemplateId(tid);
        if (CollUtil.isEmpty(configList)) {
            log.warn("模板配置为空");
            return;
        }

        SysMsgTemplateConfigDO configDO = null;
        List<SysMsgTemplateReceiverDO> receivers = null;
        for (TemplateConfigSaveVO configSaveVO : configList) {
            configDO = new SysMsgTemplateConfigDO();
            configDO.setTemplateId(tid);
            configDO.setTemplateCode(tCode);
            configDO.setTemplateName(tName);
            configDO.setSendTypeCode(configSaveVO.getSendType().name());
            configDO.setSendTypeSwitch(configSaveVO.getSendTypeSwitch());
            configDO.setSendTypeName(configSaveVO.getSendType().getDescription());
            configDO.setMesSubject(configSaveVO.getMesSubject());
            configDO.setMesTitle(configSaveVO.getMesTitle());
            configDO.setMesRichContentSign(configSaveVO.getMesRichContentSign());
            configDO.setExternalTemplateId(configSaveVO.getExternalTemplateId());
            configDO.setExternalSignName(configSaveVO.getExternalSignName());
            configDO.setMesText(configSaveVO.getMesText());
            configDO.setMesRichContentSign(Boolean.TRUE.equals(configSaveVO.getMesRichContentSign()));

            sysMsgTemplateConfigRepoProc.save(configDO);

            // 默认接收人
            receivers = this.convertReceiver2DO(configSaveVO.getDefaultReceiver());
            if (CollUtil.isNotEmpty(receivers)) {
                for (SysMsgTemplateReceiverDO receiver : receivers) {
                    receiver.setTemplateId(tid);
                    receiver.setTemplateConfigId(configDO.getId());
                }
                sysMsgTemplateReceiverRepoProc.save(receivers);
            }
        }
    }

    private List<SysMsgTemplateReceiverDO> convertReceiver2DO(DefaultReceiverVO receiverVO) {
        if (receiverVO == null) {
            return Collections.emptyList();
        }

        List<SysMsgTemplateReceiverDO> receivers = new ArrayList<>();
        if (CollUtil.isNotEmpty(receiverVO.getUsers())) {
            // 用户账号
            int i = 0;
            for (var user : receiverVO.getUsers()) {
                if (user.getId() == null) {
                    continue;
                }
                SysMsgTemplateReceiverDO receiverDO = new SysMsgTemplateReceiverDO();
                receiverDO.setReceiverType(MsgTemplateReceiverEnum.USER.name());
                receiverDO.setReceiverId(user.getId());
                receiverDO.setSortNo(i++);
                receivers.add(receiverDO);
            }
        }

        if (CollUtil.isNotEmpty(receiverVO.getRoles())) {
            // 角色
            int i = 0;
            for (var role : receiverVO.getRoles()) {
                if (role.getId() == null) {
                    continue;
                }
                SysMsgTemplateReceiverDO receiverDO = new SysMsgTemplateReceiverDO();
                receiverDO.setReceiverType(MsgTemplateReceiverEnum.ROLE.name());
                receiverDO.setReceiverId(role.getId());
                receiverDO.setSortNo(i++);
                receivers.add(receiverDO);
            }
        }
        return receivers;
    }

    private List<SysMsgTemplateConfigVO> getTemplateConfigList(SysMsgTemplateDO sysMsgTemplateDO) {
        var configList = sysMsgTemplateConfigRepoProc.listByTemplateId(sysMsgTemplateDO.getId());
        if (configList.isEmpty()) {
            return Collections.emptyList();
        }

        // 默认接收人
        var receiverMap = new HashMap<>(this.getTemplateReceiver(sysMsgTemplateDO.getId()));

        return configList.stream().map(t -> {
            var config = SysMsgTemplateConfigConvert.INSTANCE.doToVo(t);
            config.setDefaultReceiver(receiverMap.computeIfAbsent(t.getId(), configId -> emptyDefaultReceiver()));
            return config;
        }).collect(Collectors.toList());
    }

    private DefaultReceiverVO emptyDefaultReceiver() {
        DefaultReceiverVO defaultReceiverVO = new DefaultReceiverVO();
        defaultReceiverVO.setUsers(Collections.emptyList());
        defaultReceiverVO.setRoles(Collections.emptyList());

        return defaultReceiverVO;
    }

    private Map<Long, DefaultReceiverVO> getTemplateReceiver(long templateId) {
        var receiverDoList = sysMsgTemplateReceiverRepoProc.listByTemplateId(templateId);
        if (receiverDoList.isEmpty()) {
            return Collections.emptyMap();
        }

        // 用户姓名查询
        var userIds = receiverDoList.stream()
                .filter(t -> MsgTemplateReceiverEnum.USER.name().equals(t.getReceiverType()))
                .map(MsgTemplateReceiverBO::getReceiverId)
                .collect(Collectors.toSet());
        Map<Long, IdNameParam> userMap = userIds.isEmpty() ? Collections.emptyMap() : userQueryService.queryUserName(userIds).computeData()
                .stream().collect(Collectors.toMap(IdNameParam::getId, t -> t, (t1, t2) -> t1));
        // 角色信息查询
        var roleIds = receiverDoList.stream()
                .filter(t -> MsgTemplateReceiverEnum.ROLE.name().equals(t.getReceiverType()))
                .map(MsgTemplateReceiverBO::getReceiverId)
                .collect(Collectors.toSet());
        Map<Long, IdNameParam> roleMap = roleIds.isEmpty() ? Collections.emptyMap() : roleQueryService.listCodeNamesById(roleIds).computeData()
                .stream().map(t -> new IdNameParam(t.getId(), t.getName())).collect(Collectors.toMap(IdNameParam::getId, t -> t, (t1, t2) -> t1));
        // 员工查询
        var employeeIds = receiverDoList.stream()
                .filter(t -> MsgTemplateReceiverEnum.EMPLOYEE.name().equals(t.getReceiverType()))
                .map(MsgTemplateReceiverBO::getReceiverId)
                .collect(Collectors.toSet());
        Map<Long, IdNameParam> employeeMap = employeeIds.isEmpty() ? Collections.emptyMap() : employeeQueryService.queryCodeName(employeeIds).computeData()
                .stream().map(t -> new IdNameParam(t.getId(), t.getName())).collect(Collectors.toMap(IdNameParam::getId, t -> t, (t1, t2) -> t1));

        return receiverDoList.stream()
                .collect(Collectors.groupingBy(MsgTemplateReceiverBO::getTemplateConfigId,
                        Collectors.collectingAndThen(Collectors.toList(),
                                receivers -> {
                                    DefaultReceiverVO defaultReceiverVO = new DefaultReceiverVO();
                                    defaultReceiverVO.setUsers(new ArrayList<>());
                                    defaultReceiverVO.setRoles(new ArrayList<>());

                                    for (MsgTemplateReceiverBO receiverDO : receivers) {
                                        if (MsgTemplateReceiverEnum.USER.name().equals(receiverDO.getReceiverType())) {
                                            defaultReceiverVO.getUsers().add(new IdNameParam(receiverDO.getReceiverId(), userMap.computeIfAbsent(receiverDO.getReceiverId(), k -> new IdNameParam()).getName()));
                                            continue;
                                        }
                                        if (MsgTemplateReceiverEnum.ROLE.name().equals(receiverDO.getReceiverType())) {
                                            defaultReceiverVO.getRoles().add(new IdNameParam(receiverDO.getReceiverId(), roleMap.computeIfAbsent(receiverDO.getReceiverId(), k -> new IdNameParam()).getName()));
                                            continue;
                                        }
                                        log.error("不支持的类型：{}", receiverDO.getReceiverType());
                                    }

                                    return defaultReceiverVO;
                                }))
                );
    }

    private Set<Long> getTemplateReceiverUserIds(long templateId, long configId) {
        var receiverDoList = sysMsgTemplateReceiverRepoProc.listByTemplateIdAndConfigId(templateId, configId);
        if (receiverDoList.isEmpty()) {
            return Collections.emptySet();
        }

        Set<Long> userIds = new HashSet<>();
        Set<Long> employeeIds = new HashSet<>();
        Set<Long> roleIds = new HashSet<>();
        for (MsgTemplateReceiverBO receiverBO : receiverDoList) {
            if (MsgTemplateReceiverEnum.USER.name().equals(receiverBO.getReceiverType())) {
                userIds.add(receiverBO.getReceiverId());
                continue;
            }
            if (MsgTemplateReceiverEnum.EMPLOYEE.name().equals(receiverBO.getReceiverType())) {
                employeeIds.add(receiverBO.getReceiverId());
                continue;
            }
            if (MsgTemplateReceiverEnum.ROLE.name().equals(receiverBO.getReceiverType())) {
                roleIds.add(receiverBO.getReceiverId());
                continue;
            }
        }

        // 员工ID
        if (!employeeIds.isEmpty()) {
            var employeeUserIds = employeeRepoProc.getUserIds(employeeIds).values();
            userIds.addAll(employeeUserIds);
        }
        // 角色ID
        if (!roleIds.isEmpty()) {
            var roleUserIds = userRoleRepoProc.getUsersIdByRole(roleIds);
            userIds.addAll(roleUserIds);
        }

        return userIds;
    }
}