package com.elitescloud.cloudt.system.provider.export;

import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.excel.common.DataExport;
import com.elitescloud.boot.util.DatetimeUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.convert.EmployeeConvert;
import com.elitescloud.cloudt.system.model.vo.query.org.EmployeePageQueryVO;
import com.elitescloud.cloudt.system.provider.export.param.ExportEmployeeBO;
import com.elitescloud.cloudt.system.service.EmployeeMngService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.Collections;
import java.util.stream.Collectors;

/**
 * 员工导出.
 *
 * @author Kaiser（wang shao）
 * @date 2/21/2023
 */
@Slf4j
@Component
public class EmployeeExportProvider implements DataExport<ExportEmployeeBO, EmployeePageQueryVO> {

    private final EmployeeMngService service;

    public EmployeeExportProvider(EmployeeMngService service) {
        this.service = service;
    }

    @Override
    public String getTmplCode() {
        return "system_employee_export";
    }

    @Override
    public PagingVO<ExportEmployeeBO> executeExport(EmployeePageQueryVO queryParam) {
        var pageData = service.page(queryParam).getData()
                .map(EmployeeConvert.INSTANCE::pageRespVo2ExportBO);
        if (pageData.isEmpty()) {
            return pageData;
        }

        var empIds = pageData.stream().map(ExportEmployeeBO::getId).collect(Collectors.toList());
        var userIds = pageData.stream().map(ExportEmployeeBO::getUserId).collect(Collectors.toList());
        var empOrgMap = service.getEmployeeOrgs(empIds).getData();
        var empRoleMap = service.getEmployeeRolesByUserIds(userIds).getData();

        pageData.each(t -> {
            if (t.getJoinTime() != null) {
                t.setJoinTimeStr(DatetimeUtil.FORMATTER_DATETIME.format(t.getJoinTime()));
            }
            t.setEnabledName(Boolean.FALSE.equals(t.getEnabled()) ? "否" : "是");

            // 组织信息
            if (empOrgMap != null) {
                t.setOrgNames(empOrgMap.getOrDefault(t.getId(), Collections.emptyList()).stream().map(IdCodeNameParam::getName).collect(Collectors.joining(",")));
                t.setOrgCodes(empOrgMap.getOrDefault(t.getId(), Collections.emptyList()).stream().map(IdCodeNameParam::getCode).collect(Collectors.joining(",")));
            }
            // 角色信息
            if (empRoleMap != null) {
                t.setRoleNames(empRoleMap.getOrDefault(t.getUserId(), Collections.emptyList()).stream().map(IdCodeNameParam::getName).collect(Collectors.joining(",")));
                t.setRoleCodes(empRoleMap.getOrDefault(t.getUserId(), Collections.emptyList()).stream().map(IdCodeNameParam::getCode).collect(Collectors.joining(",")));
            }
        });

        // 查询员工组织
        return pageData;
    }

    @Override
    public Integer pageSize() {
        return 10;
    }
}
