package com.elitescloud.cloudt.system.provider.imports;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.constant.Gender;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.excel.common.DataImport;
import com.elitescloud.boot.util.DatetimeUtil;
import com.elitescloud.cloudt.system.constant.EmployeeType;
import com.elitescloud.cloudt.system.dto.SysUdcDTO;
import com.elitescloud.cloudt.system.provider.imports.param.ImportEmployeeBO;
import com.elitescloud.cloudt.system.service.EmployeeMngService;
import com.elitescloud.cloudt.system.service.OrgQueryService;
import com.elitescloud.cloudt.system.service.RoleQueryService;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 员工导入.
 *
 * @author Kaiser（wang shao）
 * @date 2/21/2023
 */
@Component
@Log4j2
public class EmployeeImportProvider implements DataImport<ImportEmployeeBO> {

    private UdcProvider udcProvider;
    private EmployeeMngService mngService;
    private OrgQueryService orgQueryService;
    private RoleQueryService roleQueryService;

    @Override
    public String getTmplCode() {
        return "system_employee_import";
    }

    @Override
    public List<String> executeImport(List<ImportEmployeeBO> dataList, int startRowIndex) {
        // UDC值
        var udcNameValueMap = udcProvider.listByUdcCode(Set.of(new EmployeeType().getUdcCode())).stream()
                .collect(Collectors.toMap(SysUdcDTO::getUdcCode,
                        t -> t.getValueList().stream().collect(Collectors.toMap(SysUdcDTO.UdcValue::getUdcValueName, SysUdcDTO.UdcValue::getUdcValueCode, (t1, t2) -> t1)),
                        (t1, t2) -> t1));
        // 组织编号与ID
        Map<String, Long> orgCodeIdMap = new HashMap<>(32);
        // 角色编号与ID
        Map<String, Long> roleCodeIdMap = new HashMap<>(32);
        // 性别
        Map<String, String> genderMap = Gender.all().stream().collect(Collectors.toMap(Gender::getDescription, Gender::getValue, (t1, t2) -> t1));

        // 开始导入
        List<String> failMsg = new ArrayList<>();
        for (ImportEmployeeBO employeeBO : dataList) {
            String errMsg = null;
            try {
                errMsg = this.saveEmployee(employeeBO, udcNameValueMap, orgCodeIdMap, roleCodeIdMap, genderMap);
            } catch (Exception e) {
                log.error("导入员工异常：", e);
                errMsg = e.getMessage();
            }

            failMsg.add(errMsg);
        }

        // 返回结果
        return failMsg;
    }

    @Override
    public Integer stepSize() {
        return 10;
    }

    private String saveEmployee(ImportEmployeeBO employeeBO, Map<String, Map<String, String>> udcMap, Map<String, Long> orgCodeIdMap,
                                Map<String, Long> roleCodeIdMap, Map<String, String> genderMap) {
        Assert.hasText(employeeBO.getCode(), "员工编码为空");
        Assert.hasText(employeeBO.getUsername(), "登录号为空");
        Assert.hasText(employeeBO.getFullName(), "姓名为空");
        // 性别
        if (StringUtils.hasText(employeeBO.getGenderName())) {
            employeeBO.setGender(genderMap.get(employeeBO.getGenderName()));
        }

        // 类型
        if (StringUtils.hasText(employeeBO.getTypeName())) {
            var typeMap = udcMap.computeIfAbsent(new EmployeeType().getUdcCode(), udcCode -> {
                var udc = udcProvider.getByUdcCode(udcCode);
                if (udc == null || CollUtil.isEmpty(udc.getValueList())) {
                    return Collections.emptyMap();
                }

                return udc.getValueList().stream().collect(Collectors.toMap(SysUdcDTO.UdcValue::getUdcValueName,
                        SysUdcDTO.UdcValue::getUdcValueCode, (t1, t2) -> t1));
            });
            employeeBO.setType(typeMap.get(employeeBO.getTypeName()));
        }

        employeeBO.setEnabled("是".equals(employeeBO.getEnabledName()));
        if (StringUtils.hasText(employeeBO.getJoinTimeStr())) {
            try {
                employeeBO.setJoinTime(LocalDateTime.parse(employeeBO.getJoinTimeStr(), DatetimeUtil.FORMATTER_DATETIME));
            } catch (Exception e) {
                throw new IllegalArgumentException("时间格式不正确");
            }
        }

        // 组织
        if (StringUtils.hasText(employeeBO.getOrgCodes())) {
            Set<Long> orgIds = new HashSet<>();
            for (String s : employeeBO.getOrgCodes().split(",")) {
                var orgId = orgCodeIdMap.computeIfAbsent(s.trim(), t -> {
                    var id = orgQueryService.getIdByOrgCode(t).getData();
                    Assert.notNull(id, "组织[" + t + "]不存在");
                    return id;
                });
                orgIds.add(orgId);
            }
            employeeBO.setOrgIds(orgIds);
        }
        Assert.notEmpty(employeeBO.getOrgIds(), "未配置有效组织");

        // 角色
        if (StringUtils.hasText(employeeBO.getRoleCodes())) {
            Set<Long> roleIds = new HashSet<>();
            for (String s : employeeBO.getRoleCodes().split(",")) {
                var roleId = roleCodeIdMap.computeIfAbsent(s.trim(), t -> {
                    var id = roleQueryService.getIdByCode(t).getData();
                    Assert.notNull(id, "角色[" + t + "]不存在");
                    return id;
                });
                roleIds.add(roleId);
            }
            employeeBO.setRoleIds(roleIds);
        }

        // 保存
        var saveResult = mngService.importEmployee(employeeBO);
        return saveResult.isSuccess() ? null : saveResult.getMsg();
    }

    @Autowired
    public void setUdcProvider(UdcProvider udcProvider) {
        this.udcProvider = udcProvider;
    }

    @Autowired
    public void setMngService(EmployeeMngService mngService) {
        this.mngService = mngService;
    }

    @Autowired
    public void setOrgQueryService(OrgQueryService orgQueryService) {
        this.orgQueryService = orgQueryService;
    }

    @Autowired
    public void setRoleQueryService(RoleQueryService roleQueryService) {
        this.roleQueryService = roleQueryService;
    }
}
