package com.elitescloud.cloudt.system.rest;

import com.elitescloud.cloudt.common.base.ApiResult;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.bootstrap.encrypt.AbstractEnvironmentDecrypt;
import org.springframework.http.MediaType;
import org.springframework.security.crypto.encrypt.TextEncryptor;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.constraints.NotBlank;

/**
 * 系统工具类
 *
 * @author Michael Li
 * 2020/6/21
 */
@RestController
@RequestMapping(value = "/mng/sys/support", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
@Api(tags = "系统安全")
public class SystemSupportController {
    private static final String ENCRYPTOR_PREFIX = AbstractEnvironmentDecrypt.ENCRYPTED_PROPERTY_PREFIX;

    @Autowired(required = false)
    private TextEncryptor textEncryptor;

    /**
     * 加密字符串
     *
     * @param plainText 明文
     * @return 密文
     */
    @GetMapping(value = "/encrypt")
    @ApiImplicitParam(name = "plainText", value = "加密内容", required = true)
    @ApiOperation("加密字符串")
    public ApiResult<String> encrypt(@NotBlank(message = "加密内容不能为空") @RequestParam(name = "plainText") String plainText) {
        if (textEncryptor == null) {
            return ApiResult.fail("加密失败，请联系管理员");
        }

        String ciphertext = textEncryptor.encrypt(plainText);
        return ApiResult.ok(ENCRYPTOR_PREFIX + ciphertext);
    }

    /**
     * 解密字符串
     *
     * @param ciphertext 密文
     * @return 密文
     */
    @ApiOperation("解密字符串")
    @PostMapping(value = "/decrypt")
    public ApiResult<String> decrypt(@RequestBody String ciphertext) {
        if (textEncryptor == null) {
            return ApiResult.fail("解密失败，请联系管理员");
        }

        if (ciphertext.startsWith(ENCRYPTOR_PREFIX)) {
            ciphertext = ciphertext.substring(ENCRYPTOR_PREFIX.length());
        }
        String plainText = textEncryptor.decrypt(ciphertext);
        return ApiResult.ok(plainText);
    }
}
