package com.elitescloud.cloudt.system.rpc;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.util.EncryptUtils;
import com.elitescloud.cloudt.system.constant.UserSourceType;
import com.elitescloud.cloudt.system.param.*;
import com.elitescloud.cloudt.system.provider.SysUserRpcService;
import com.elitescloud.cloudt.system.provider.dto.SysUserRpcDTO;
import com.elitescloud.cloudt.system.provider.param.SysUserRpcDtoParam;
import com.elitescloud.cloudt.system.provider.param.SysUserRpcSaveParam;
import com.elitescloud.cloudt.system.service.UserMngService;
import com.elitescloud.cloudt.system.service.UserQueryService;
import com.elitescloud.cloudt.system.service.old.ISysUserService;
import com.elitescloud.cloudt.system.vo.SysPermissionVO;
import com.elitescloud.cloudt.system.vo.SysRoleVO;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.elitescloud.cloudt.system.vo.SysUserVO;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.io.UnsupportedEncodingException;
import java.util.Collections;
import java.util.List;
import java.util.Set;

/**
 * @author Chris
 * 2021-08-27
 */
@RestController
@RequestMapping(value = SysUserRpcService.URI, produces = MediaType.APPLICATION_JSON_VALUE)
public class SysUserRpcServiceImpl implements SysUserRpcService {

    private ISysUserService sysUserService;
    private UserMngService userMngService;
    private UserQueryService userQueryService;

    /**
     * 根据查询条件检索用户信息
     *
     * @param param 查询条件对象
     * @return 用户信息集合
     */
    @Override
    public PagingVO<SysUserVO> search(SysUserQueryParam param) {
        return sysUserService.search(param);
    }

    /**
     * @param id
     * @return
     */
    @Override
    public SysUserDTO getById(Long id) {
        return sysUserService.getById(id);
    }

    @Override
    public List<SysUserVO> getByIds(List<Long> ids) {
        return sysUserService.getByIds(ids);
    }

    /**
     * 创建新用户
     *
     * @param user 用户账号信息
     * @return 新用户账号ID
     */
    @Override
    public Long create(SysUserNewParam user) {
        return sysUserService.create(user, null);
    }

    /**
     * 创建新用户
     *
     * @param user 用户账号信息
     * @return 新用户账号ID
     */
    @Override
    public Long create(SysUserNewParam user, String password) {
        return sysUserService.create(user, password);
    }

    @Override
    public void update(SysUserUpdateParam user) {
        sysUserService.update(user);
    }

    @Override
    public Boolean switchUserStatus(Long id) {
        return sysUserService.switchUserStatus(id);
    }

    /**
     * 用户 批量启用/禁用
     *
     * @param param 参数
     */
    @Override
    public void batchSwitchUserStatus(SysUserBatchSwitchParam param) {
        sysUserService.batchSwitchUserStatus(param);
    }

    @Override
    public Set<SysRoleVO> listRolesByUsername(String username) {
        return sysUserService.listRolesByUsername(username);
    }

    @Override
    public SysUserDTO current() {
        return sysUserService.current();
    }

    @Override
    public List<SysPermissionVO> currentMenu() {
        return Collections.emptyList();
    }

    @Override
    public Set<SysPermissionVO> currentActions() {
        return Collections.emptySet();
    }

    @Override
    public Set<SysPermissionVO> currentMenuActions(Long menuId) {
        return Collections.emptySet();
    }

    @Override
    public void deleteBatch(List<Long> idList) {
        sysUserService.deleteBatch(idList);
    }

    @Override
    public void updatePassword(SysPasswordUpdate update) {
        userMngService.updatePassword(update);
    }

    @Override
    public void changePasswordNoSec(Long userId, String pwd) {
        userMngService.updatePassword(userId, pwd);
    }

    @Override
    public ApiResult<String> updateUserWechatOpenid(String userName, String openid) {
        return sysUserService.updateUserWechatOpenid(userName, openid);
    }

    @Override
    public List<SysUserRpcDTO> findUserRpcDtoByParam(SysUserRpcDtoParam param) {
        return sysUserService.findUserRpcDtoByParam(param);
    }

    @Override
    public ApiResult<Object> sysUserAccountCreate(SysUserRpcSaveParam param) {

        if (StringUtils.isBlank(param.getUsername())) {
            return ApiResult.fail(ApiCode.BUSINESS_EXCEPTION, "用户名不能为空");
        }
        if (StringUtils.isBlank(param.getPassword())) {
            return ApiResult.fail(ApiCode.FAIL, "密码不能为空");
        }

        SysUserNewParam newUser = new SysUserNewParam();
        newUser.setUsername(param.getUsername());
        newUser.setFirstName(param.getFirstName());
        newUser.setLastName(param.getLastName());
        newUser.setMobile(param.getMobile());
        newUser.setEmail(param.getEmail());
        newUser.setEnabled(param.getEnabled());
        newUser.setOuId(param.getOuId());
        newUser.setOuCode(param.getOuCode());
        newUser.setOuName(param.getOuName());
        newUser.setRoleIds(param.getRoleIds());
        newUser.setDataRoleIds(param.getDataRoleIds());
        newUser.setFlowRoleIds(param.getFlowRoleIds());
        newUser.setSourceType(CharSequenceUtil.blankToDefault(param.getSourceType(), UserSourceType.OTHER.getValue()));
        newUser.setTerminals(param.getTerminals());
        newUser.setTenantId(param.getTenantId());

        var userId = sysUserService.create(newUser, param.getPassword());

        return ApiResult.ok(userId);
    }

    @Override
    public ApiResult<Long> sysUserCreate(SysUserNewParam user, String password) {
        return ApiResult.ok(sysUserService.create(user, password));
    }

    @Override
    public ApiResult<Long> getUserIdByUsername(String username) {
        var userId = sysUserService.getUserIdByUsername(username);
        return ApiResult.ok(userId);
    }

    @Override
    public ApiResult<SysUserDTO> getUserByUsername(String username) {
        var user = sysUserService.getUserByUsername(username);
        return ApiResult.ok(user);
    }

    @Override
    public ApiResult<SysUserDTO> getUserByMobile(String mobile) {
        var user = sysUserService.getUserByMobile(mobile);
        return ApiResult.ok(user);
    }

    @Override
    public ApiResult<SysUserDTO> getUserByEmail(String email) {
        var user = sysUserService.getUserByEmail(email);
        return ApiResult.ok(user);
    }

    @Override
    public ApiResult<SysUserDTO> getUserById(String userId) {
        var user = sysUserService.getUserById(Long.parseLong(userId));
        return ApiResult.ok(user);
    }

    @Override
    public ApiResult<SysUserDTO> getUserByWechatOpenid(String wechatOpenid) {
        var user = sysUserService.getUserByWechatOpenid(wechatOpenid);
        return ApiResult.ok(user);
    }

    @Override
    public ApiResult<SysUserDTO> getUserByAccount(String account) {
        var user = sysUserService.getUserByAccount(account);
        return ApiResult.ok(user);
    }

    @Override
    public Boolean existsByUserName(String userName) {
        return userQueryService.existsUsername(userName).getData();
    }

    @Override
    public Boolean existsByUserMobile(String mobile) {
        return userQueryService.existsMobile(mobile).getData();
    }

    @Autowired
    public void setSysUserService(ISysUserService sysUserService) {
        this.sysUserService = sysUserService;
    }

    @Autowired
    public void setUserMngService(UserMngService userMngService) {
        this.userMngService = userMngService;
    }

    @Autowired
    public void setUserQueryService(UserQueryService userQueryService) {
        this.userQueryService = userQueryService;
    }
}
